/**
 * AtoShip Admin Scripts
 *
 * @package AtoShip
 */

(function($) {
    'use strict';

    var AtoShipAdmin = {
        init: function() {
            this.bindEvents();
        },

        bindEvents: function() {
            $('#atoship-test-connection').on('click', this.testConnection);
            $('.atoship-copy-btn').on('click', this.copyToClipboard);
            $('#atoship-regenerate-secret').on('click', this.regenerateSecret);
            $('#atoship-disconnect-oauth').on('click', this.disconnectOAuth);
        },

        testConnection: function(e) {
            e.preventDefault();

            var $btn = $(this);
            var $spinner = $btn.siblings('.spinner');
            var $status = $('#atoship-connection-status');

            $btn.prop('disabled', true);
            $spinner.addClass('is-active');
            $status.html('<span class="atoship-status atoship-status-unknown">' + atoshipAdmin.i18n.testing + '</span>');

            $.ajax({
                url: atoshipAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'atoship_test_connection',
                    nonce: atoshipAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $status.html('<span class="atoship-status atoship-status-connected">' + atoshipAdmin.i18n.connected + '</span>');
                        if (response.data.account && response.data.account.company_name) {
                            $status.append(' - ' + response.data.account.company_name);
                        }
                    } else {
                        $status.html('<span class="atoship-status atoship-status-error">' + atoshipAdmin.i18n.error + '</span>');
                        if (response.data && response.data.message) {
                            $status.append('<br><small>' + response.data.message + '</small>');
                        }
                    }
                },
                error: function() {
                    $status.html('<span class="atoship-status atoship-status-error">' + atoshipAdmin.i18n.error + '</span>');
                },
                complete: function() {
                    $btn.prop('disabled', false);
                    $spinner.removeClass('is-active');
                }
            });
        },

        copyToClipboard: function(e) {
            e.preventDefault();

            var $btn = $(this);
            var targetId = $btn.data('copy');
            var $target = $('#' + targetId);
            var text = $target.text();

            if (navigator.clipboard && navigator.clipboard.writeText) {
                navigator.clipboard.writeText(text).then(function() {
                    AtoShipAdmin.showCopyFeedback($btn);
                });
            } else {
                // Fallback for older browsers
                var $temp = $('<input>');
                $('body').append($temp);
                $temp.val(text).select();
                document.execCommand('copy');
                $temp.remove();
                AtoShipAdmin.showCopyFeedback($btn);
            }
        },

        showCopyFeedback: function($btn) {
            var originalText = $btn.text();
            $btn.text(atoshipAdmin.i18n.copied);
            setTimeout(function() {
                $btn.text(originalText);
            }, 2000);
        },

        regenerateSecret: function(e) {
            e.preventDefault();

            if (!confirm('Are you sure you want to regenerate the webhook secret? You will need to update it in your AtoShip dashboard.')) {
                return;
            }

            var $btn = $(this);
            $btn.prop('disabled', true);

            $.ajax({
                url: atoshipAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'atoship_regenerate_secret',
                    nonce: atoshipAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $('#atoship-webhook-secret').text(response.data.secret);
                    } else {
                        alert('Failed to regenerate secret.');
                    }
                },
                complete: function() {
                    $btn.prop('disabled', false);
                }
            });
        },

        disconnectOAuth: function(e) {
            e.preventDefault();

            if (!confirm('Are you sure you want to disconnect your AtoShip account? You will need to reconnect to use the integration.')) {
                return;
            }

            var $btn = $(this);
            $btn.prop('disabled', true).text('Disconnecting...');

            $.ajax({
                url: atoshipAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'atoship_disconnect_oauth',
                    nonce: atoshipAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert('Failed to disconnect: ' + (response.data && response.data.message ? response.data.message : 'Unknown error'));
                        $btn.prop('disabled', false).text('Disconnect OAuth');
                    }
                },
                error: function() {
                    alert('Failed to disconnect. Please try again.');
                    $btn.prop('disabled', false).text('Disconnect OAuth');
                }
            });
        }
    };

    $(document).ready(function() {
        AtoShipAdmin.init();
    });

})(jQuery);
