/**
 * AtoShip Label Metabox JavaScript
 *
 * Handles AJAX interactions for the shipping label meta box
 * on WooCommerce order edit pages.
 *
 * Rates are displayed for reference only.
 * Label purchasing is done on the AtoShip dashboard.
 */
(function ($) {
    'use strict';

    var AtoShipLabel = {

        init: function () {
            $(document)
                .on('click', '.atoship-get-rates', this.getRates.bind(this))
                .on('click', '.atoship-void-label', this.voidLabel.bind(this));
        },

        /**
         * Get rates from AtoShip API (for reference/comparison)
         */
        getRates: function (e) {
            e.preventDefault();
            var $btn = $(e.currentTarget);
            var orderId = $btn.data('order-id');
            var $form = $btn.closest('.atoship-rate-form');

            this.showLoading($form, atoshipLabel.i18n.fetching_rates);
            $form.find('.atoship-error').hide();
            $form.find('.atoship-rates-container').hide();

            $.ajax({
                url: atoshipLabel.ajax_url,
                type: 'POST',
                data: {
                    action: 'atoship_get_rates',
                    nonce: atoshipLabel.nonce,
                    order_id: orderId
                },
                success: function (response) {
                    AtoShipLabel.hideLoading($form);

                    if (response.success && response.data.rates) {
                        AtoShipLabel.renderRates($form, response.data.rates);
                    } else {
                        var msg = (response.data && response.data.message)
                            ? response.data.message
                            : atoshipLabel.i18n.no_rates;
                        AtoShipLabel.showError($form, msg);
                    }
                },
                error: function () {
                    AtoShipLabel.hideLoading($form);
                    AtoShipLabel.showError($form, atoshipLabel.i18n.error);
                }
            });
        },

        /**
         * Render rates list (read-only reference)
         */
        renderRates: function ($form, rates) {
            var $container = $form.find('.atoship-rates-container');
            var $list = $form.find('.atoship-rates-list');
            $list.empty();

            if (!rates || rates.length === 0) {
                this.showError($form, atoshipLabel.i18n.no_rates);
                return;
            }

            // Sort by price
            rates.sort(function (a, b) {
                return a.rate - b.rate;
            });

            for (var i = 0; i < rates.length; i++) {
                var rate = rates[i];
                var deliveryText = '';
                if (rate.delivery_days) {
                    deliveryText = rate.delivery_days + ' ' + atoshipLabel.i18n.business_days;
                }

                var $row = $(
                    '<div class="atoship-rate-row atoship-rate-readonly">' +
                        '<span class="atoship-rate-info">' +
                            '<span class="atoship-rate-service">' +
                                this.escHtml(rate.carrier) + ' ' + this.escHtml(rate.service) +
                            '</span>' +
                            (deliveryText ? '<span class="atoship-rate-days">' + this.escHtml(deliveryText) + '</span>' : '') +
                        '</span>' +
                        '<span class="atoship-rate-price">$' + parseFloat(rate.rate).toFixed(2) + '</span>' +
                    '</div>'
                );

                $list.append($row);
            }

            $container.show();
        },

        /**
         * Void label
         */
        voidLabel: function (e) {
            e.preventDefault();

            if (!confirm(atoshipLabel.i18n.confirm_void)) {
                return;
            }

            var $btn = $(e.currentTarget);
            var orderId = $btn.data('order-id');
            var labelId = $btn.data('label-id');

            $btn.prop('disabled', true).text(atoshipLabel.i18n.voiding_label);

            $.ajax({
                url: atoshipLabel.ajax_url,
                type: 'POST',
                data: {
                    action: 'atoship_void_label',
                    nonce: atoshipLabel.nonce,
                    order_id: orderId,
                    label_id: labelId
                },
                success: function (response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        $btn.prop('disabled', false).text('Void Label');
                        alert((response.data && response.data.message) || atoshipLabel.i18n.error);
                    }
                },
                error: function () {
                    $btn.prop('disabled', false).text('Void Label');
                    alert(atoshipLabel.i18n.error);
                }
            });
        },

        /**
         * Show loading state
         */
        showLoading: function ($form, text) {
            var $loading = $form.find('.atoship-loading');
            $loading.find('.atoship-loading-text').text(text);
            $loading.show();
            $form.find('.atoship-actions button').prop('disabled', true);
        },

        /**
         * Hide loading state
         */
        hideLoading: function ($form) {
            $form.find('.atoship-loading').hide();
            $form.find('.atoship-actions button').prop('disabled', false);
        },

        /**
         * Show error message
         */
        showError: function ($form, message) {
            $form.find('.atoship-error')
                .html('<p>' + this.escHtml(message) + '</p>')
                .show();
        },

        /**
         * Escape HTML
         */
        escHtml: function (str) {
            if (!str) return '';
            var div = document.createElement('div');
            div.appendChild(document.createTextNode(str));
            return div.innerHTML;
        }
    };

    $(document).ready(function () {
        AtoShipLabel.init();
    });

})(jQuery);
