<?php
/**
 * Plugin Name: AtoShip for WooCommerce
 * Plugin URI: https://atoship.com/woocommerce
 * Description: Connect your WooCommerce store to AtoShip for streamlined shipping management, label printing, and order fulfillment.
 * Version: 1.0.0
 * Author: AtoShip
 * Author URI: https://atoship.com
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain: atoship-for-woocommerce
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 6.0
 * WC tested up to: 8.5
 *
 * @package AtoShip
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Plugin constants
define( 'ATOSHIP_VERSION', '1.0.0' );
define( 'ATOSHIP_PLUGIN_FILE', __FILE__ );
define( 'ATOSHIP_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'ATOSHIP_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

// Configurable via wp-config.php: define('ATOSHIP_DEV_URL', 'http://192.168.x.x:3001')
if ( ! defined( 'ATOSHIP_API_BASE_URL' ) ) {
    define( 'ATOSHIP_API_BASE_URL', 'https://atoship.com/api/v1' );
}
if ( ! defined( 'ATOSHIP_APP_URL' ) ) {
    define( 'ATOSHIP_APP_URL', 'https://atoship.com' );
}
if ( ! defined( 'ATOSHIP_OAUTH_CLIENT_ID' ) ) {
    define( 'ATOSHIP_OAUTH_CLIENT_ID', 'cmkvlt19s0001w0mmmz84glcp' );
}

/**
 * Check if WooCommerce is active
 */
function atoship_check_woocommerce() {
    if ( ! class_exists( 'WooCommerce' ) ) {
        add_action( 'admin_notices', 'atoship_woocommerce_missing_notice' );
        return false;
    }
    return true;
}

/**
 * WooCommerce missing notice
 */
function atoship_woocommerce_missing_notice() {
    ?>
    <div class="notice notice-error">
        <p><?php esc_html_e( 'AtoShip for WooCommerce requires WooCommerce to be installed and activated.', 'atoship-for-woocommerce' ); ?></p>
    </div>
    <?php
}

/**
 * Initialize the plugin
 */
function atoship_init() {
    if ( ! atoship_check_woocommerce() ) {
        return;
    }

    // Load plugin classes
    require_once ATOSHIP_PLUGIN_DIR . 'includes/class-atoship-oauth.php';
    require_once ATOSHIP_PLUGIN_DIR . 'includes/class-atoship-api.php';
    require_once ATOSHIP_PLUGIN_DIR . 'includes/class-atoship-settings.php';
    require_once ATOSHIP_PLUGIN_DIR . 'includes/class-atoship-order-sync.php';
    require_once ATOSHIP_PLUGIN_DIR . 'includes/class-atoship-webhook.php';
    require_once ATOSHIP_PLUGIN_DIR . 'includes/class-atoship-order-actions.php';
    require_once ATOSHIP_PLUGIN_DIR . 'includes/class-atoship-shipping-method.php';
    require_once ATOSHIP_PLUGIN_DIR . 'includes/class-atoship-label-metabox.php';

    // Initialize components
    AtoShip_OAuth::init();
    AtoShip_Settings::init();
    AtoShip_Order_Sync::init();
    AtoShip_Webhook::init();
    AtoShip_Order_Actions::init();
    AtoShip_Label_Metabox::init();
}
add_action( 'plugins_loaded', 'atoship_init' );

/**
 * Add "AtoShip - Discount Labels" bulk action to orders list.
 * Redirects selected orders to the AtoShip dashboard for label purchasing.
 */
function atoship_register_bulk_actions( $bulk_actions ) {
    $bulk_actions['atoship_discount_labels'] = __( 'AtoShip - Buy Labels', 'atoship-for-woocommerce' );
    return $bulk_actions;
}
add_filter( 'bulk_actions-edit-shop_order', 'atoship_register_bulk_actions' );
add_filter( 'bulk_actions-woocommerce_page_wc-orders', 'atoship_register_bulk_actions' );

/**
 * Handle the AtoShip bulk action — redirect to AtoShip dashboard.
 */
function atoship_handle_bulk_action( $redirect_to, $action, $post_ids ) {
    if ( 'atoship_discount_labels' !== $action ) {
        return $redirect_to;
    }

    $dashboard_url = rtrim( ATOSHIP_APP_URL, '/' ) . '/dashboard/orders';

    // Pass order IDs as query parameter so AtoShip can pre-filter
    if ( ! empty( $post_ids ) ) {
        $dashboard_url = add_query_arg( 'wc_orders', implode( ',', array_map( 'absint', $post_ids ) ), $dashboard_url );
    }

    wp_redirect( $dashboard_url );
    exit;
}
add_filter( 'handle_bulk_actions-edit-shop_order', 'atoship_handle_bulk_action', 10, 3 );
add_filter( 'handle_bulk_actions-woocommerce_page_wc-orders', 'atoship_handle_bulk_action', 10, 3 );

/**
 * Add plugin action links (shown on Plugins page)
 *
 * @param array $links Existing links
 * @return array
 */
function atoship_plugin_action_links( $links ) {
    $delete_url = wp_nonce_url(
        admin_url( 'admin.php?action=atoship_delete_plugin' ),
        'atoship_delete_plugin'
    );
    $links['delete'] = '<a href="' . esc_url( $delete_url ) . '" style="color: #b32d2e;" onclick="return confirm(\'' . esc_js( __( 'Are you sure? This will disconnect from AtoShip and remove the plugin.', 'atoship-for-woocommerce' ) ) . '\');">' . __( 'Delete', 'atoship-for-woocommerce' ) . '</a>';

    return $links;
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'atoship_plugin_action_links' );

/**
 * Handle plugin deletion action
 *
 * Disconnects from AtoShip, deactivates plugin, then redirects to WP's delete confirmation.
 */
function atoship_handle_delete_plugin() {
    check_admin_referer( 'atoship_delete_plugin' );

    if ( ! current_user_can( 'delete_plugins' ) ) {
        wp_die( esc_html__( 'You do not have permission to delete plugins.', 'atoship-for-woocommerce' ) );
    }

    // Disconnect from AtoShip (revoke token + clear local data)
    if ( class_exists( 'AtoShip_OAuth' ) ) {
        AtoShip_OAuth::disconnect();
    }

    $plugin_file = plugin_basename( ATOSHIP_PLUGIN_FILE );

    // Deactivate the plugin first (required before WP allows deletion)
    deactivate_plugins( $plugin_file );

    // Redirect to WordPress's built-in delete confirmation page
    $delete_url = wp_nonce_url(
        admin_url( 'plugins.php?action=delete-selected&checked[]=' . urlencode( $plugin_file ) . '&plugin_status=inactive' ),
        'bulk-plugins'
    );

    wp_redirect( $delete_url );
    exit;
}
add_action( 'admin_action_atoship_delete_plugin', 'atoship_handle_delete_plugin' );

/**
 * Activation hook
 */
function atoship_activate() {
    // Create database tables if needed
    global $wpdb;

    $table_name = $wpdb->prefix . 'atoship_shipments';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        order_id bigint(20) NOT NULL,
        atoship_shipment_id varchar(100) NOT NULL,
        carrier varchar(50) DEFAULT NULL,
        tracking_number varchar(100) DEFAULT NULL,
        tracking_url varchar(500) DEFAULT NULL,
        label_url varchar(500) DEFAULT NULL,
        status varchar(50) DEFAULT 'pending',
        ship_date datetime DEFAULT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY order_id (order_id),
        KEY atoship_shipment_id (atoship_shipment_id)
    ) $charset_collate;";

    require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
    dbDelta( $sql );

    // Set default options
    add_option( 'atoship_api_key', '' );
    add_option( 'atoship_webhook_secret', wp_generate_password( 32, false ) );
    add_option( 'atoship_auto_sync', 'yes' );
    add_option( 'atoship_sync_statuses', array( 'processing', 'on-hold' ) );
    add_option( 'atoship_debug_log', 'no' );

    // Flush rewrite rules for webhook endpoint
    flush_rewrite_rules();
}
register_activation_hook( __FILE__, 'atoship_activate' );

/**
 * Deactivation hook
 */
function atoship_deactivate() {
    flush_rewrite_rules();
}
register_deactivation_hook( __FILE__, 'atoship_deactivate' );

/**
 * Declare HPOS compatibility
 */
add_action( 'before_woocommerce_init', function() {
    if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
    }
} );
