<?php
/**
 * AtoShip API Client
 *
 * Handles all communication with the AtoShip API
 * Based on AtoShip API documentation
 *
 * @package AtoShip
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AtoShip_API {

    /**
     * API Key (sk_live_* or sk_test_*)
     *
     * @var string
     */
    private $api_key;

    /**
     * API Base URL
     *
     * @var string
     */
    private $api_base_url;

    /**
     * Is test mode
     *
     * @var bool
     */
    private $is_test_mode;

    /**
     * Singleton instance
     *
     * @var AtoShip_API
     */
    private static $instance = null;

    /**
     * Get singleton instance
     *
     * @return AtoShip_API
     */
    public static function instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->api_key      = get_option( 'atoship_api_key', '' );
        $this->api_base_url = ATOSHIP_API_BASE_URL;
        $this->is_test_mode = $this->detect_test_mode();
    }

    /**
     * Detect if using test mode based on API key prefix
     *
     * @return bool
     */
    private function detect_test_mode() {
        return strpos( $this->api_key, 'sk_test_' ) === 0;
    }

    /**
     * Check if API is configured (either OAuth or API Key)
     *
     * @return bool
     */
    public function is_configured() {
        return AtoShip_OAuth::is_connected();
    }

    /**
     * Check if API key format is valid
     *
     * @return bool
     */
    public function is_valid_api_key() {
        return preg_match( '/^sk_(live|test)_[a-zA-Z0-9]+$/', $this->api_key );
    }

    /**
     * Check if in test mode
     *
     * @return bool
     */
    public function is_test_mode() {
        return $this->is_test_mode;
    }

    /**
     * Make API request
     *
     * @param string $endpoint API endpoint
     * @param string $method   HTTP method
     * @param array  $data     Request data
     * @return array|WP_Error
     */
    public function request( $endpoint, $method = 'GET', $data = array() ) {
        if ( ! $this->is_configured() ) {
            return new WP_Error(
                'atoship_not_configured',
                __( 'AtoShip is not connected. Please connect your store in AtoShip settings.', 'atoship-for-woocommerce' )
            );
        }

        $url = trailingslashit( $this->api_base_url ) . ltrim( $endpoint, '/' );

        // Get OAuth access token
        $auth_token = AtoShip_OAuth::get_access_token();
        if ( ! $auth_token ) {
            return new WP_Error( 'atoship_no_token', __( 'No valid OAuth token. Please reconnect your store.', 'atoship-for-woocommerce' ) );
        }

        $args = array(
            'method'  => $method,
            'timeout' => 60,
            'headers' => array(
                'Authorization' => 'Bearer ' . $auth_token,
                'Content-Type'  => 'application/json',
                'Accept'        => 'application/json',
                'User-Agent'    => 'AtoShip-WooCommerce/' . ATOSHIP_VERSION,
            ),
        );

        if ( ! empty( $data ) && in_array( $method, array( 'POST', 'PUT', 'PATCH' ), true ) ) {
            $args['body'] = wp_json_encode( $data );
        }

        if ( ! empty( $data ) && 'GET' === $method ) {
            $url = add_query_arg( $data, $url );
        }

        $this->log( "API Request: $method $endpoint", 'debug' );

        $response = wp_remote_request( $url, $args );

        if ( is_wp_error( $response ) ) {
            $this->log( 'API request failed: ' . $response->get_error_message(), 'error' );
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );
        $decoded_body  = json_decode( $response_body, true );

        // Handle specific error codes
        if ( 401 === $response_code ) {
            $this->log( 'API authentication failed: Invalid or revoked API key', 'error' );
            return new WP_Error(
                'atoship_unauthorized',
                __( 'Invalid or revoked API key. Please check your API key in AtoShip dashboard.', 'atoship-for-woocommerce' ),
                array( 'status' => 401 )
            );
        }

        if ( 403 === $response_code ) {
            $this->log( 'API forbidden: Insufficient permissions', 'error' );
            return new WP_Error(
                'atoship_forbidden',
                __( 'Your API key does not have permission for this action.', 'atoship-for-woocommerce' ),
                array( 'status' => 403 )
            );
        }

        if ( $response_code >= 400 ) {
            // API returns { "error": { "code": "...", "message": "..." } } or { "message": "..." }
            $error_message = 'Unknown error';
            if ( isset( $decoded_body['error']['message'] ) ) {
                $error_message = $decoded_body['error']['message'];
            } elseif ( isset( $decoded_body['error'] ) && is_string( $decoded_body['error'] ) ) {
                $error_message = $decoded_body['error'];
            } elseif ( isset( $decoded_body['message'] ) ) {
                $error_message = $decoded_body['message'];
            }
            $this->log( "API error ($response_code): $error_message", 'error' );
            return new WP_Error( 'atoship_api_error', $error_message, array( 'status' => $response_code ) );
        }

        return $decoded_body;
    }

    /**
     * Test API connection
     *
     * @return array|WP_Error
     */
    public function test_connection() {
        return $this->request( 'account' );
    }

    /**
     * Get account information
     *
     * @return array|WP_Error
     */
    public function get_account() {
        return $this->request( 'account' );
    }

    /**
     * ========================================
     * Order Operations
     * ========================================
     */

    /**
     * Create/sync order to AtoShip
     *
     * @param WC_Order $order WooCommerce order
     * @return array|WP_Error
     */
    public function create_order( $order ) {
        $order_data = $this->format_order_data( $order );
        return $this->request( 'orders', 'POST', $order_data );
    }

    /**
     * Create multiple orders (batch)
     *
     * @param array $orders Array of WC_Order objects
     * @return array|WP_Error
     */
    public function create_orders_batch( $orders ) {
        $orders_data = array();
        foreach ( $orders as $order ) {
            $orders_data[] = $this->format_order_data( $order );
        }
        return $this->request( 'orders/batch', 'POST', array( 'orders' => $orders_data ) );
    }

    /**
     * Get order from AtoShip
     *
     * @param string $order_id AtoShip order ID or external_id
     * @return array|WP_Error
     */
    public function get_order( $order_id ) {
        return $this->request( 'orders/' . $order_id );
    }

    /**
     * List orders from AtoShip
     *
     * @param array $params Query parameters (page, limit, status, etc.)
     * @return array|WP_Error
     */
    public function list_orders( $params = array() ) {
        return $this->request( 'orders', 'GET', $params );
    }

    /**
     * ========================================
     * Label Operations
     * ========================================
     */

    /**
     * Create shipping label
     *
     * @param array $label_data Label data
     * @return array|WP_Error
     */
    public function create_label( $label_data ) {
        return $this->request( 'labels', 'POST', $label_data );
    }

    /**
     * Get label details
     *
     * @param string $label_id Label ID
     * @return array|WP_Error
     */
    public function get_label( $label_id ) {
        return $this->request( 'labels/' . $label_id );
    }

    /**
     * Purchase a draft label
     *
     * @param string $label_id Label ID
     * @return array|WP_Error
     */
    public function purchase_label( $label_id ) {
        return $this->request( 'labels/' . $label_id . '/purchase', 'POST' );
    }

    /**
     * Void/cancel a label
     *
     * @param string $label_id Label ID
     * @return array|WP_Error
     */
    public function void_label( $label_id ) {
        return $this->request( 'labels/' . $label_id . '/void', 'POST' );
    }

    /**
     * ========================================
     * Tracking Operations
     * ========================================
     */

    /**
     * Get tracking information
     *
     * @param string $tracking_number Tracking number
     * @param string $carrier         Carrier code (optional)
     * @return array|WP_Error
     */
    public function get_tracking( $tracking_number, $carrier = '' ) {
        $endpoint = 'tracking/' . $tracking_number;
        $params   = array();
        if ( ! empty( $carrier ) ) {
            $params['carrier'] = $carrier;
        }
        return $this->request( $endpoint, 'GET', $params );
    }

    /**
     * ========================================
     * Carrier & Rates
     * ========================================
     */

    /**
     * Get available carriers
     *
     * @return array|WP_Error
     */
    public function get_carriers() {
        return $this->request( 'carriers' );
    }

    /**
     * Get shipping rates
     *
     * @param array $shipment_data Shipment data for rate calculation
     * @return array|WP_Error
     */
    public function get_rates( $shipment_data ) {
        return $this->request( 'rates', 'POST', $shipment_data );
    }

    /**
     * ========================================
     * Address Operations
     * ========================================
     */

    /**
     * Validate address
     *
     * @param array $address Address data
     * @return array|WP_Error
     */
    public function validate_address( $address ) {
        return $this->request( 'addresses/validate', 'POST', $address );
    }

    /**
     * ========================================
     * Webhook Management
     * ========================================
     */

    /**
     * Register webhook endpoint
     *
     * @param string $url    Webhook URL
     * @param array  $events Events to subscribe to
     * @return array|WP_Error
     */
    public function register_webhook( $url, $events = array() ) {
        $default_events = array(
            'label.created',
            'label.purchased',
            'label.voided',
            'tracking.created',
            'tracking.updated',
            'tracking.delivered',
            'tracking.exception',
        );

        return $this->request( 'webhooks', 'POST', array(
            'url'    => $url,
            'events' => ! empty( $events ) ? $events : $default_events,
            'active' => true,
        ) );
    }

    /**
     * ========================================
     * Helper Methods
     * ========================================
     */

    /**
     * Format WooCommerce order data for AtoShip API
     *
     * @param WC_Order $order WooCommerce order
     * @return array
     */
    public function format_order_data( $order ) {
        $items = array();
        foreach ( $order->get_items() as $item ) {
            $product = $item->get_product();
            $item_data = array(
                'name'        => $item->get_name(),
                'sku'         => $product ? $product->get_sku() : '',
                'quantity'    => $item->get_quantity(),
                'unit_price'  => (float) ( $item->get_total() / $item->get_quantity() ),
                'total_price' => (float) $item->get_total(),
                'currency'    => $order->get_currency(),
            );

            if ( $product ) {
                $item_data['weight'] = (float) $product->get_weight();
                $item_data['dimensions'] = array(
                    'length' => (float) $product->get_length(),
                    'width'  => (float) $product->get_width(),
                    'height' => (float) $product->get_height(),
                );

                // Customs info for international shipping
                $hs_code = $product->get_meta( '_hs_code' );
                $origin_country = $product->get_meta( '_origin_country' );
                if ( $hs_code ) {
                    $item_data['hs_code'] = $hs_code;
                }
                if ( $origin_country ) {
                    $item_data['origin_country'] = $origin_country;
                }
            }

            $items[] = $item_data;
        }

        return array(
            'external_id'     => (string) $order->get_id(),
            'external_number' => $order->get_order_number(),
            'external_source' => 'woocommerce',
            'external_url'    => $order->get_edit_order_url(),
            'created_at'      => $order->get_date_created()->format( 'c' ),
            'status'          => $order->get_status(),
            'currency'        => $order->get_currency(),
            'subtotal'        => (float) $order->get_subtotal(),
            'shipping_cost'   => (float) $order->get_shipping_total(),
            'tax_total'       => (float) $order->get_total_tax(),
            'discount_total'  => (float) $order->get_discount_total(),
            'total'           => (float) $order->get_total(),
            'shipping_method' => $order->get_shipping_method(),
            'payment_method'  => $order->get_payment_method_title(),
            'customer_note'   => $order->get_customer_note(),
            'ship_to'         => array(
                'name'       => trim( $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name() ),
                'first_name' => $order->get_shipping_first_name(),
                'last_name'  => $order->get_shipping_last_name(),
                'company'    => $order->get_shipping_company(),
                'address1'   => $order->get_shipping_address_1(),
                'address2'   => $order->get_shipping_address_2(),
                'city'       => $order->get_shipping_city(),
                'state'      => $order->get_shipping_state(),
                'postcode'   => $order->get_shipping_postcode(),
                'country'    => $order->get_shipping_country(),
                'phone'      => $order->get_billing_phone(),
                'email'      => $order->get_billing_email(),
            ),
            'bill_to'         => array(
                'name'       => trim( $order->get_billing_first_name() . ' ' . $order->get_billing_last_name() ),
                'first_name' => $order->get_billing_first_name(),
                'last_name'  => $order->get_billing_last_name(),
                'company'    => $order->get_billing_company(),
                'address1'   => $order->get_billing_address_1(),
                'address2'   => $order->get_billing_address_2(),
                'city'       => $order->get_billing_city(),
                'state'      => $order->get_billing_state(),
                'postcode'   => $order->get_billing_postcode(),
                'country'    => $order->get_billing_country(),
                'phone'      => $order->get_billing_phone(),
                'email'      => $order->get_billing_email(),
            ),
            'items'           => $items,
            'weight_unit'     => get_option( 'woocommerce_weight_unit', 'kg' ),
            'dimension_unit'  => get_option( 'woocommerce_dimension_unit', 'cm' ),
        );
    }

    /**
     * Verify webhook signature (HMAC-SHA256)
     *
     * @param string $payload   Raw request body
     * @param string $signature Signature from X-Atoship-Signature header
     * @param string $secret    Webhook secret
     * @return bool
     */
    public static function verify_webhook_signature( $payload, $signature, $secret ) {
        $expected_signature = hash_hmac( 'sha256', $payload, $secret );
        return hash_equals( $expected_signature, $signature );
    }

    /**
     * Log message using WooCommerce logger
     *
     * @param string $message Log message
     * @param string $level   Log level (debug, info, notice, warning, error, critical, alert, emergency)
     */
    private function log( $message, $level = 'info' ) {
        if ( function_exists( 'wc_get_logger' ) && ( 'yes' === get_option( 'atoship_debug_log', 'no' ) || 'error' === $level ) ) {
            $logger = wc_get_logger();
            $logger->log( $level, $message, array( 'source' => 'atoship' ) );
        }
    }
}
