<?php
/**
 * AtoShip Label Meta Box
 *
 * Adds a shipping label meta box to the WooCommerce order edit page.
 * Allows merchants to get rates, buy labels, view tracking, and void labels
 * directly from WooCommerce admin.
 *
 * @package AtoShip
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AtoShip_Label_Metabox {

    /**
     * Initialize hooks
     */
    public static function init() {
        add_action( 'add_meta_boxes', array( __CLASS__, 'add_meta_box' ) );
        add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_scripts' ) );

        // AJAX handlers
        add_action( 'wp_ajax_atoship_get_rates', array( __CLASS__, 'ajax_get_rates' ) );
        add_action( 'wp_ajax_atoship_void_label', array( __CLASS__, 'ajax_void_label' ) );
    }

    /**
     * Get AtoShip dashboard URL
     *
     * @param string $path Optional path to append (e.g., '/dashboard/orders')
     * @return string
     */
    public static function get_app_url( $path = '' ) {
        $base = ATOSHIP_APP_URL;
        return rtrim( $base, '/' ) . '/' . ltrim( $path, '/' );
    }

    /**
     * Register meta box for both HPOS and legacy order screens
     */
    public static function add_meta_box() {
        $screen = self::get_order_screen_id();

        add_meta_box(
            'atoship-label-metabox',
            __( 'AtoShip Shipping', 'atoship-for-woocommerce' ),
            array( __CLASS__, 'render_meta_box' ),
            $screen,
            'side',
            'high'
        );
    }

    /**
     * Get the order edit screen ID (HPOS compatible)
     *
     * @return string
     */
    private static function get_order_screen_id() {
        if ( class_exists( '\Automattic\WooCommerce\Utilities\OrderUtil' ) &&
             \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled() ) {
            return wc_get_page_screen_id( 'shop-order' );
        }
        return 'shop_order';
    }

    /**
     * Get order from post or HPOS
     *
     * @param WP_Post|null $post
     * @return WC_Order|false
     */
    private static function get_order_from_screen( $post ) {
        if ( class_exists( '\Automattic\WooCommerce\Utilities\OrderUtil' ) &&
             \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled() ) {
            // HPOS: get order ID from URL
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $order_id = isset( $_GET['id'] ) ? absint( $_GET['id'] ) : 0;
            if ( $order_id ) {
                return wc_get_order( $order_id );
            }
        }

        if ( $post instanceof WP_Post ) {
            return wc_get_order( $post->ID );
        }

        return false;
    }

    /**
     * Enqueue scripts for the meta box
     *
     * @param string $hook Current admin page hook
     */
    public static function enqueue_scripts( $hook ) {
        $screen = get_current_screen();
        if ( ! $screen ) {
            return;
        }

        $order_screen = self::get_order_screen_id();
        if ( $screen->id !== $order_screen && 'shop_order' !== $screen->id ) {
            return;
        }

        wp_enqueue_style(
            'atoship-admin-css',
            ATOSHIP_PLUGIN_URL . 'admin/css/atoship-admin.css',
            array(),
            ATOSHIP_VERSION
        );

        wp_enqueue_script(
            'atoship-label-metabox',
            ATOSHIP_PLUGIN_URL . 'admin/js/atoship-label-metabox.js',
            array( 'jquery' ),
            ATOSHIP_VERSION,
            true
        );

        wp_localize_script( 'atoship-label-metabox', 'atoshipLabel', array(
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce'    => wp_create_nonce( 'atoship_label' ),
            'i18n'     => array(
                'fetching_rates'  => __( 'Fetching rates...', 'atoship-for-woocommerce' ),
                'voiding_label'   => __( 'Voiding label...', 'atoship-for-woocommerce' ),
                'no_rates'        => __( 'No rates available for this shipment.', 'atoship-for-woocommerce' ),
                'error'           => __( 'An error occurred. Please try again.', 'atoship-for-woocommerce' ),
                'confirm_void'    => __( 'Are you sure you want to void this label? This cannot be undone.', 'atoship-for-woocommerce' ),
                'business_days'   => __( 'business days', 'atoship-for-woocommerce' ),
            ),
        ) );
    }

    /**
     * Render the meta box content
     *
     * @param WP_Post $post
     */
    public static function render_meta_box( $post ) {
        $order = self::get_order_from_screen( $post );
        if ( ! $order ) {
            echo '<p>' . esc_html__( 'Order not found.', 'atoship-for-woocommerce' ) . '</p>';
            return;
        }

        $order_id = $order->get_id();

        // Check API configuration
        $api = AtoShip_API::instance();
        if ( ! $api->is_configured() ) {
            echo '<p class="atoship-notice atoship-notice-warning">';
            echo esc_html__( 'AtoShip is not configured. ', 'atoship-for-woocommerce' );
            echo '<a href="' . esc_url( admin_url( 'admin.php?page=wc-settings&tab=atoship' ) ) . '">';
            echo esc_html__( 'Configure now', 'atoship-for-woocommerce' );
            echo '</a></p>';
            return;
        }

        // Check if label already exists
        $existing_label = self::get_existing_label( $order );

        if ( $existing_label ) {
            self::render_label_info( $order, $existing_label );
        } else {
            self::render_rate_form( $order );
        }
    }

    /**
     * Get existing label data for an order
     *
     * @param WC_Order $order
     * @return array|false Label data or false
     */
    private static function get_existing_label( $order ) {
        global $wpdb;
        $table = $wpdb->prefix . 'atoship_shipments';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        $shipment = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM $table WHERE order_id = %d AND status != 'voided' ORDER BY created_at DESC LIMIT 1",
                $order->get_id()
            ),
            ARRAY_A
        );

        if ( $shipment ) {
            return $shipment;
        }

        // Also check order meta for labels from webhook
        $tracking_number = $order->get_meta( '_atoship_tracking_number' );
        if ( $tracking_number ) {
            return array(
                'tracking_number' => $tracking_number,
                'carrier'         => $order->get_meta( '_atoship_carrier' ) ?: '',
                'label_url'       => $order->get_meta( '_atoship_label_url' ) ?: '',
                'tracking_url'    => $order->get_meta( '_atoship_tracking_url' ) ?: '',
                'status'          => $order->get_meta( '_atoship_label_status' ) ?: 'purchased',
                'atoship_shipment_id' => $order->get_meta( '_atoship_shipment_id' ) ?: '',
            );
        }

        return false;
    }

    /**
     * Render label information when a label already exists
     *
     * @param WC_Order $order
     * @param array    $label
     */
    private static function render_label_info( $order, $label ) {
        $order_id = $order->get_id();
        $status = $label['status'] ?? 'purchased';
        ?>
        <div class="atoship-label-info" data-order-id="<?php echo esc_attr( $order_id ); ?>">
            <div class="atoship-label-status atoship-status-<?php echo esc_attr( $status ); ?>">
                <span class="atoship-status-badge"><?php echo esc_html( ucfirst( $status ) ); ?></span>
            </div>

            <?php if ( ! empty( $label['carrier'] ) ) : ?>
                <div class="atoship-label-row">
                    <strong><?php esc_html_e( 'Carrier:', 'atoship-for-woocommerce' ); ?></strong>
                    <span><?php echo esc_html( $label['carrier'] ); ?></span>
                </div>
            <?php endif; ?>

            <?php if ( ! empty( $label['tracking_number'] ) ) : ?>
                <div class="atoship-label-row">
                    <strong><?php esc_html_e( 'Tracking:', 'atoship-for-woocommerce' ); ?></strong>
                    <?php if ( ! empty( $label['tracking_url'] ) ) : ?>
                        <a href="<?php echo esc_url( $label['tracking_url'] ); ?>" target="_blank" rel="noopener">
                            <?php echo esc_html( $label['tracking_number'] ); ?>
                        </a>
                    <?php else : ?>
                        <span><?php echo esc_html( $label['tracking_number'] ); ?></span>
                    <?php endif; ?>
                </div>
            <?php endif; ?>

            <div class="atoship-label-actions">
                <?php if ( ! empty( $label['label_url'] ) ) : ?>
                    <a href="<?php echo esc_url( $label['label_url'] ); ?>"
                       class="button button-primary"
                       target="_blank" rel="noopener">
                        <?php esc_html_e( 'View Label', 'atoship-for-woocommerce' ); ?>
                    </a>
                <?php endif; ?>

                <?php if ( 'purchased' === $status && ! empty( $label['atoship_shipment_id'] ) ) : ?>
                    <button type="button"
                            class="button atoship-void-label"
                            data-order-id="<?php echo esc_attr( $order_id ); ?>"
                            data-label-id="<?php echo esc_attr( $label['atoship_shipment_id'] ); ?>">
                        <?php esc_html_e( 'Void Label', 'atoship-for-woocommerce' ); ?>
                    </button>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    /**
     * Render the rate selection form when no label exists
     *
     * @param WC_Order $order
     */
    private static function render_rate_form( $order ) {
        $order_id = $order->get_id();

        // Build package info from order
        $package_info = self::get_order_package_info( $order );
        ?>
        <div class="atoship-rate-form" data-order-id="<?php echo esc_attr( $order_id ); ?>">

            <!-- Package Info Summary -->
            <div class="atoship-package-info">
                <div class="atoship-label-row">
                    <strong><?php esc_html_e( 'From:', 'atoship-for-woocommerce' ); ?></strong>
                    <span><?php echo esc_html( $package_info['from_summary'] ); ?></span>
                </div>
                <div class="atoship-label-row">
                    <strong><?php esc_html_e( 'To:', 'atoship-for-woocommerce' ); ?></strong>
                    <span><?php echo esc_html( $package_info['to_summary'] ); ?></span>
                </div>
                <div class="atoship-label-row">
                    <strong><?php esc_html_e( 'Package:', 'atoship-for-woocommerce' ); ?></strong>
                    <span><?php echo esc_html( $package_info['package_summary'] ); ?></span>
                </div>
            </div>

            <!-- Rates Container -->
            <div class="atoship-rates-container" style="display:none;">
                <div class="atoship-rates-list"></div>
            </div>

            <!-- Error Container -->
            <div class="atoship-error" style="display:none;"></div>

            <!-- Loading Spinner -->
            <div class="atoship-loading" style="display:none;">
                <span class="spinner is-active"></span>
                <span class="atoship-loading-text"></span>
            </div>

            <!-- Action Buttons -->
            <div class="atoship-actions">
                <button type="button"
                        class="button atoship-get-rates"
                        data-order-id="<?php echo esc_attr( $order_id ); ?>">
                    <?php esc_html_e( 'Get Rates', 'atoship-for-woocommerce' ); ?>
                </button>

                <a href="<?php echo esc_url( self::get_app_url( 'dashboard/orders' ) ); ?>"
                   class="button button-primary"
                   target="_blank" rel="noopener">
                    <?php esc_html_e( 'Buy on AtoShip', 'atoship-for-woocommerce' ); ?>
                    <span class="dashicons dashicons-external" style="line-height:inherit;font-size:14px;margin-left:2px;"></span>
                </a>
            </div>
        </div>
        <?php
    }

    /**
     * Get order package info for display
     *
     * @param WC_Order $order
     * @return array
     */
    private static function get_order_package_info( $order ) {
        // From address (store address)
        $country_state = get_option( 'woocommerce_default_country', '' );
        $parts = explode( ':', $country_state );
        $store_state = $parts[1] ?? '';
        $store_zip   = get_option( 'woocommerce_store_postcode', '' );
        $store_city  = get_option( 'woocommerce_store_city', '' );

        // To address
        $to_city    = $order->get_shipping_city();
        $to_state   = $order->get_shipping_state();
        $to_zip     = $order->get_shipping_postcode();
        $to_country = $order->get_shipping_country();

        // Package weight/dimensions from items
        $total_weight = 0;
        $wc_weight_unit = get_option( 'woocommerce_weight_unit', 'kg' );

        foreach ( $order->get_items() as $item ) {
            $product = $item->get_product();
            if ( $product ) {
                $weight = (float) $product->get_weight();
                if ( $weight > 0 ) {
                    $total_weight += $weight * $item->get_quantity();
                }
            }
        }

        $weight_display = $total_weight > 0
            ? round( $total_weight, 2 ) . ' ' . $wc_weight_unit
            : __( 'Not set', 'atoship-for-woocommerce' );

        return array(
            'from_summary'    => sprintf( '%s, %s %s', $store_city, $store_state, $store_zip ),
            'to_summary'      => sprintf( '%s, %s %s %s', $to_city, $to_state, $to_zip, $to_country ),
            'package_summary' => $weight_display,
        );
    }

    /**
     * AJAX: Get rates for an order
     */
    public static function ajax_get_rates() {
        check_ajax_referer( 'atoship_label', 'nonce' );

        if ( ! current_user_can( 'edit_shop_orders' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'atoship-for-woocommerce' ) ) );
        }

        $order_id = isset( $_POST['order_id'] ) ? absint( $_POST['order_id'] ) : 0;
        $order    = wc_get_order( $order_id );

        if ( ! $order ) {
            wp_send_json_error( array( 'message' => __( 'Order not found.', 'atoship-for-woocommerce' ) ) );
        }

        $api = AtoShip_API::instance();

        // Build from address
        $country_state = get_option( 'woocommerce_default_country', '' );
        $parts = explode( ':', $country_state );

        $from_address = array(
            'name'    => get_bloginfo( 'name' ),
            'street1' => get_option( 'woocommerce_store_address', '' ),
            'street2' => get_option( 'woocommerce_store_address_2', '' ),
            'city'    => get_option( 'woocommerce_store_city', '' ),
            'state'   => $parts[1] ?? '',
            'zip'     => get_option( 'woocommerce_store_postcode', '' ),
            'country' => $parts[0] ?? 'US',
        );

        // Build to address from order
        $to_address = array(
            'name'    => trim( $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name() ),
            'company' => $order->get_shipping_company(),
            'street1' => $order->get_shipping_address_1(),
            'street2' => $order->get_shipping_address_2(),
            'city'    => $order->get_shipping_city(),
            'state'   => $order->get_shipping_state(),
            'zip'     => $order->get_shipping_postcode(),
            'country' => $order->get_shipping_country() ?: 'US',
            'phone'   => $order->get_billing_phone(),
        );

        // Build parcel from order items
        $parcel = self::build_parcel_from_order( $order );

        if ( ! $parcel ) {
            wp_send_json_error( array(
                'message' => __( 'Cannot calculate rates: order items have no weight. Please set product weights.', 'atoship-for-woocommerce' ),
            ) );
        }

        $request_data = array(
            'from_address' => $from_address,
            'to_address'   => $to_address,
            'parcel'       => $parcel,
            'skip_address_validation' => true,
        );

        $response = $api->get_rates( $request_data );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( array( 'message' => $response->get_error_message() ) );
        }

        if ( isset( $response['object'] ) && 'AddressValidation' === $response['object'] ) {
            wp_send_json_error( array(
                'message' => __( 'Address validation issue. Please check the shipping address.', 'atoship-for-woocommerce' ),
            ) );
        }

        if ( ! isset( $response['data'] ) || empty( $response['data'] ) ) {
            wp_send_json_error( array( 'message' => __( 'No rates available for this shipment.', 'atoship-for-woocommerce' ) ) );
        }

        // Format rates for frontend
        $rates = array();
        foreach ( $response['data'] as $rate ) {
            $rates[] = array(
                'id'            => $rate['id'] ?? '',
                'carrier'       => $rate['carrier'] ?? 'Unknown',
                'service'       => $rate['service'] ?? 'Standard',
                'service_code'  => $rate['service_code'] ?? '',
                'rate'          => floatval( $rate['rate'] ?? 0 ),
                'currency'      => $rate['currency'] ?? 'USD',
                'delivery_days' => isset( $rate['delivery_days'] ) ? intval( $rate['delivery_days'] ) : null,
                'list_rate'     => floatval( $rate['list_rate'] ?? 0 ),
            );
        }

        wp_send_json_success( array(
            'rates'        => $rates,
            'from_address' => $from_address,
            'to_address'   => $to_address,
            'parcel'       => $parcel,
        ) );
    }

    /**
     * AJAX: Void a label
     */
    public static function ajax_void_label() {
        check_ajax_referer( 'atoship_label', 'nonce' );

        if ( ! current_user_can( 'edit_shop_orders' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'atoship-for-woocommerce' ) ) );
        }

        $order_id = isset( $_POST['order_id'] ) ? absint( $_POST['order_id'] ) : 0;
        $label_id = isset( $_POST['label_id'] ) ? sanitize_text_field( wp_unslash( $_POST['label_id'] ) ) : '';

        if ( empty( $label_id ) ) {
            wp_send_json_error( array( 'message' => __( 'Label ID is required.', 'atoship-for-woocommerce' ) ) );
        }

        $order = wc_get_order( $order_id );
        if ( ! $order ) {
            wp_send_json_error( array( 'message' => __( 'Order not found.', 'atoship-for-woocommerce' ) ) );
        }

        $api = AtoShip_API::instance();
        $response = $api->void_label( $label_id );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( array( 'message' => $response->get_error_message() ) );
        }

        // Update order meta
        $order->update_meta_data( '_atoship_label_status', 'voided' );
        $order->save();
        $order->add_order_note( __( 'AtoShip label voided.', 'atoship-for-woocommerce' ) );

        // Update shipments table
        global $wpdb;
        $table = $wpdb->prefix . 'atoship_shipments';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        $wpdb->update(
            $table,
            array( 'status' => 'voided' ),
            array( 'atoship_shipment_id' => $label_id ),
            array( '%s' ),
            array( '%s' )
        );

        wp_send_json_success( array( 'message' => __( 'Label voided successfully.', 'atoship-for-woocommerce' ) ) );
    }

    /**
     * Build parcel data from order items
     *
     * @param WC_Order $order
     * @return array|false
     */
    private static function build_parcel_from_order( $order ) {
        $total_weight  = 0;
        $max_length    = 0;
        $max_width     = 0;
        $max_height    = 0;
        $has_dims      = false;

        $wc_weight_unit = get_option( 'woocommerce_weight_unit', 'kg' );
        $wc_dim_unit    = get_option( 'woocommerce_dimension_unit', 'cm' );

        foreach ( $order->get_items() as $item ) {
            $product = $item->get_product();
            if ( ! $product ) {
                continue;
            }

            $weight = (float) $product->get_weight();
            $qty    = $item->get_quantity();
            if ( $weight > 0 ) {
                $total_weight += $weight * $qty;
            }

            $l = (float) $product->get_length();
            $w = (float) $product->get_width();
            $h = (float) $product->get_height();
            if ( $l > 0 && $w > 0 && $h > 0 ) {
                $has_dims = true;
                $max_length = max( $max_length, $l );
                $max_width  = max( $max_width, $w );
                $max_height = max( $max_height, $h * $qty );
            }
        }

        if ( $total_weight <= 0 ) {
            // Default to 1 lb per item when no weight is configured
            $default_weight_per_item = ( 'oz' === $wc_weight_unit ) ? 16 : 1;
            if ( 'kg' === $wc_weight_unit ) {
                $default_weight_per_item = 0.45;
            } elseif ( 'g' === $wc_weight_unit ) {
                $default_weight_per_item = 450;
            }
            $total_qty = 0;
            foreach ( $order->get_items() as $item ) {
                $total_qty += $item->get_quantity();
            }
            $total_weight = $default_weight_per_item * max( $total_qty, 1 );
        }

        // Convert to oz / inches
        $weight_oz = self::convert_weight( $total_weight, $wc_weight_unit );

        if ( $has_dims ) {
            $length = self::convert_dim( $max_length, $wc_dim_unit );
            $width  = self::convert_dim( $max_width, $wc_dim_unit );
            $height = self::convert_dim( $max_height, $wc_dim_unit );
        } else {
            $length = 6;
            $width  = 4;
            $height = 2;
        }

        return array(
            'weight'         => round( $weight_oz, 1 ),
            'weight_unit'    => 'oz',
            'length'         => round( $length, 1 ),
            'width'          => round( $width, 1 ),
            'height'         => round( $height, 1 ),
            'dimension_unit' => 'in',
        );
    }

    /**
     * Convert weight to ounces
     */
    private static function convert_weight( $weight, $unit ) {
        switch ( strtolower( $unit ) ) {
            case 'kg':  return $weight * 35.274;
            case 'g':   return $weight * 0.035274;
            case 'lbs': return $weight * 16;
            default:    return $weight;
        }
    }

    /**
     * Convert dimension to inches
     */
    private static function convert_dim( $dim, $unit ) {
        switch ( strtolower( $unit ) ) {
            case 'cm':  return $dim * 0.3937;
            case 'm':   return $dim * 39.3701;
            case 'mm':  return $dim * 0.03937;
            default:    return $dim;
        }
    }
}
