<?php
/**
 * AtoShip OAuth Authentication
 *
 * Handles OAuth 2.0 authentication flow with AtoShip
 *
 * @package AtoShip
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AtoShip_OAuth {

    /**
     * OAuth scopes
     */
    const SCOPES = 'read:profile read:orders write:orders read:labels write:labels read:addresses read:carriers read:rates';

    /**
     * Get OAuth endpoint URL
     *
     * @param string $path OAuth path (e.g., 'authorize', 'token', 'revoke')
     * @return string
     */
    private static function get_oauth_url( $path ) {
        return rtrim( ATOSHIP_APP_URL, '/' ) . '/api/oauth/' . ltrim( $path, '/' );
    }

    /**
     * Get OAuth client ID
     *
     * @return string
     */
    private static function get_client_id() {
        return ATOSHIP_OAUTH_CLIENT_ID;
    }

    /**
     * Initialize OAuth hooks
     */
    public static function init() {
        add_action( 'admin_init', array( __CLASS__, 'handle_oauth_callback' ) );
        add_action( 'wp_ajax_atoship_disconnect_oauth', array( __CLASS__, 'ajax_disconnect_oauth' ) );
    }

    /**
     * Check if OAuth is connected
     *
     * @return bool
     */
    public static function is_connected() {
        $access_token = get_option( 'atoship_oauth_access_token', '' );
        $expires_at   = get_option( 'atoship_oauth_expires_at', 0 );

        if ( empty( $access_token ) ) {
            return false;
        }

        // Check if token is expired (with 5 minute buffer)
        if ( $expires_at > 0 && time() > ( $expires_at - 300 ) ) {
            // Try to refresh token
            return self::refresh_token();
        }

        return true;
    }

    /**
     * Get OAuth access token
     *
     * @return string|false
     */
    public static function get_access_token() {
        if ( ! self::is_connected() ) {
            return false;
        }

        return get_option( 'atoship_oauth_access_token', '' );
    }

    /**
     * Get OAuth authorization URL
     *
     * @return string
     */
    public static function get_authorize_url() {
        $redirect_uri = admin_url( 'admin.php?page=wc-settings&tab=atoship' );
        $state        = wp_create_nonce( 'atoship_oauth_state' );
        $store_url    = get_site_url();
        $store_name   = get_bloginfo( 'name' );

        // Save state for verification
        set_transient( 'atoship_oauth_state', $state, HOUR_IN_SECONDS );

        $params = array(
            'client_id'     => self::get_client_id(),
            'redirect_uri'  => $redirect_uri,
            'response_type' => 'code',
            'scope'         => self::SCOPES,
            'state'         => $state,
            'store_url'     => $store_url,
            'store_name'    => $store_name,
        );

        return self::get_oauth_url( 'authorize' ) . '?' . http_build_query( $params );
    }

    /**
     * Handle OAuth callback
     */
    public static function handle_oauth_callback() {
        // Check if this is an OAuth callback
        if ( ! isset( $_GET['page'] ) || 'wc-settings' !== $_GET['page'] ) {
            return;
        }

        if ( ! isset( $_GET['tab'] ) || 'atoship' !== $_GET['tab'] ) {
            return;
        }

        if ( ! isset( $_GET['code'] ) ) {
            return;
        }

        // Verify state to prevent CSRF
        $state          = isset( $_GET['state'] ) ? sanitize_text_field( wp_unslash( $_GET['state'] ) ) : '';
        $expected_state = get_transient( 'atoship_oauth_state' );

        if ( empty( $state ) || $state !== $expected_state ) {
            wp_die( esc_html__( 'Invalid state parameter. Please try connecting again.', 'atoship-for-woocommerce' ) );
        }

        // Delete the state transient
        delete_transient( 'atoship_oauth_state' );

        // Exchange authorization code for access token
        $code = sanitize_text_field( wp_unslash( $_GET['code'] ) );
        $result = self::exchange_code_for_token( $code );

        if ( is_wp_error( $result ) ) {
            // Show error notice
            set_transient( 'atoship_oauth_error', $result->get_error_message(), 30 );
            wp_safe_redirect( admin_url( 'admin.php?page=wc-settings&tab=atoship' ) );
            exit;
        }

        // Show success notice
        set_transient( 'atoship_oauth_success', __( 'Successfully connected to AtoShip!', 'atoship-for-woocommerce' ), 30 );

        // Redirect to clean URL
        wp_safe_redirect( admin_url( 'admin.php?page=wc-settings&tab=atoship' ) );
        exit;
    }

    /**
     * Exchange authorization code for access token
     *
     * @param string $code Authorization code
     * @return bool|WP_Error
     */
    private static function exchange_code_for_token( $code ) {
        $redirect_uri = admin_url( 'admin.php?page=wc-settings&tab=atoship' );

        $response = wp_remote_post( self::get_oauth_url( 'token' ), array(
            'body' => array(
                'grant_type'   => 'authorization_code',
                'client_id'    => self::get_client_id(),
                'code'         => $code,
                'redirect_uri' => $redirect_uri,
            ),
            'timeout' => 30,
        ) );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code( $response );
        $body        = wp_remote_retrieve_body( $response );
        $data        = json_decode( $body, true );

        if ( 200 !== $status_code ) {
            $error_message = isset( $data['error_description'] ) ? $data['error_description'] : __( 'Failed to obtain access token', 'atoship-for-woocommerce' );
            return new WP_Error( 'oauth_error', $error_message );
        }

        // Save tokens
        self::save_tokens( $data );

        return true;
    }

    /**
     * Save OAuth tokens
     *
     * @param array $token_data Token data from OAuth response
     */
    private static function save_tokens( $token_data ) {
        update_option( 'atoship_oauth_access_token', $token_data['access_token'] );
        update_option( 'atoship_oauth_refresh_token', $token_data['refresh_token'] ?? '' );
        update_option( 'atoship_oauth_token_type', $token_data['token_type'] ?? 'Bearer' );
        update_option( 'atoship_oauth_scope', $token_data['scope'] ?? self::SCOPES );

        // Calculate expiry timestamp
        $expires_in = isset( $token_data['expires_in'] ) ? (int) $token_data['expires_in'] : 3600;
        update_option( 'atoship_oauth_expires_at', time() + $expires_in );

        // Mark as OAuth authenticated
        update_option( 'atoship_auth_method', 'oauth' );
    }

    /**
     * Refresh access token
     *
     * @return bool
     */
    private static function refresh_token() {
        $refresh_token = get_option( 'atoship_oauth_refresh_token', '' );

        if ( empty( $refresh_token ) ) {
            return false;
        }

        $response = wp_remote_post( self::get_oauth_url( 'token' ), array(
            'body' => array(
                'grant_type'    => 'refresh_token',
                'client_id'     => self::get_client_id(),
                'refresh_token' => $refresh_token,
            ),
            'timeout' => 30,
        ) );

        if ( is_wp_error( $response ) ) {
            return false;
        }

        $status_code = wp_remote_retrieve_response_code( $response );
        $body        = wp_remote_retrieve_body( $response );
        $data        = json_decode( $body, true );

        if ( 200 !== $status_code ) {
            // Refresh failed, clear tokens
            self::disconnect();
            return false;
        }

        // Save new tokens
        self::save_tokens( $data );

        return true;
    }

    /**
     * Disconnect OAuth
     *
     * @param bool $revoke_remote Whether to revoke the token on the AtoShip server.
     *                            Set to false when called from the /disconnect endpoint
     *                            (atoship already revoked the token server-side).
     */
    public static function disconnect( $revoke_remote = true ) {
        if ( $revoke_remote ) {
            // Revoke token on server
            $access_token = get_option( 'atoship_oauth_access_token', '' );

            if ( ! empty( $access_token ) ) {
                wp_remote_post( self::get_oauth_url( 'revoke' ), array(
                    'body' => array(
                        'token'     => $access_token,
                        'client_id' => self::get_client_id(),
                    ),
                    'timeout' => 10,
                ) );
            }
        }

        // Clear all OAuth data
        delete_option( 'atoship_oauth_access_token' );
        delete_option( 'atoship_oauth_refresh_token' );
        delete_option( 'atoship_oauth_token_type' );
        delete_option( 'atoship_oauth_scope' );
        delete_option( 'atoship_oauth_expires_at' );
        delete_option( 'atoship_auth_method' );
    }

    /**
     * Activate connection with a pre-generated token (called by atoship backend)
     *
     * @param string $access_token  OAuth access token
     * @param string $refresh_token OAuth refresh token
     * @param int    $expires_in    Token expiry in seconds
     * @param string $scope         Token scopes
     */
    public static function activate_with_token( $access_token, $refresh_token = '', $expires_in = 3600, $scope = '' ) {
        self::save_tokens( array(
            'access_token'  => $access_token,
            'refresh_token' => $refresh_token,
            'token_type'    => 'Bearer',
            'expires_in'    => $expires_in,
            'scope'         => ! empty( $scope ) ? $scope : self::SCOPES,
        ) );
    }

    /**
     * AJAX handler for disconnecting OAuth
     */
    public static function ajax_disconnect_oauth() {
        check_ajax_referer( 'atoship_admin', 'nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'atoship-for-woocommerce' ) ) );
        }

        self::disconnect();

        wp_send_json_success( array(
            'message' => __( 'Disconnected from AtoShip.', 'atoship-for-woocommerce' ),
        ) );
    }

    /**
     * Get connected account info
     *
     * @return array|false
     */
    public static function get_account_info() {
        $info = get_transient( 'atoship_oauth_account_info' );

        if ( false !== $info ) {
            return $info;
        }

        if ( ! self::is_connected() ) {
            return false;
        }

        $access_token = self::get_access_token();

        // Make API call to get account info
        $response = wp_remote_get( ATOSHIP_API_BASE_URL . '/account', array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $access_token,
            ),
            'timeout' => 15,
        ) );

        if ( is_wp_error( $response ) ) {
            return false;
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( isset( $data['email'] ) ) {
            // Cache for 5 minutes
            set_transient( 'atoship_oauth_account_info', $data, 5 * MINUTE_IN_SECONDS );
            return $data;
        }

        return false;
    }
}
