<?php
/**
 * AtoShip Order Actions
 *
 * Adds bulk actions and meta boxes to WooCommerce orders
 *
 * @package AtoShip
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AtoShip_Order_Actions {

    /**
     * Initialize order actions
     */
    public static function init() {
        // Bulk actions
        add_filter( 'bulk_actions-woocommerce_page_wc-orders', array( __CLASS__, 'add_bulk_actions' ) );
        add_filter( 'handle_bulk_actions-woocommerce_page_wc-orders', array( __CLASS__, 'handle_bulk_actions' ), 10, 3 );

        // Legacy (CPT-based orders)
        add_filter( 'bulk_actions-edit-shop_order', array( __CLASS__, 'add_bulk_actions' ) );
        add_filter( 'handle_bulk_actions-edit-shop_order', array( __CLASS__, 'handle_bulk_actions' ), 10, 3 );

        // Admin notices for bulk actions
        add_action( 'admin_notices', array( __CLASS__, 'bulk_action_admin_notice' ) );

        // Order meta box (disabled — shipping info is in the Label Metabox)
        // add_action( 'add_meta_boxes', array( __CLASS__, 'add_order_meta_box' ) );

        // Order list columns
        add_filter( 'manage_woocommerce_page_wc-orders_columns', array( __CLASS__, 'add_order_column' ) );
        add_action( 'manage_woocommerce_page_wc-orders_custom_column', array( __CLASS__, 'render_order_column' ), 10, 2 );

        // Legacy columns
        add_filter( 'manage_edit-shop_order_columns', array( __CLASS__, 'add_order_column' ) );
        add_action( 'manage_shop_order_posts_custom_column', array( __CLASS__, 'render_order_column_legacy' ), 10, 2 );

        // Single order actions
        add_filter( 'woocommerce_order_actions', array( __CLASS__, 'add_order_actions' ) );
        add_action( 'woocommerce_order_action_atoship_sync', array( __CLASS__, 'process_order_action_sync' ) );

        // Enqueue scripts
        add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_scripts' ) );
    }

    /**
     * Add bulk actions to order list
     *
     * @param array $actions Existing bulk actions
     * @return array
     */
    public static function add_bulk_actions( $actions ) {
        $actions['atoship_sync'] = __( 'Sync to AtoShip', 'atoship-for-woocommerce' );
        return $actions;
    }

    /**
     * Handle bulk actions
     *
     * @param string $redirect_to Redirect URL
     * @param string $action      Action name
     * @param array  $order_ids   Order IDs
     * @return string
     */
    public static function handle_bulk_actions( $redirect_to, $action, $order_ids ) {
        if ( 'atoship_sync' !== $action ) {
            return $redirect_to;
        }

        $results = AtoShip_Order_Sync::sync_orders_batch( $order_ids );

        $redirect_to = add_query_arg( array(
            'atoship_synced'       => count( $results['success'] ),
            'atoship_sync_failed'  => count( $results['failed'] ),
        ), $redirect_to );

        return $redirect_to;
    }

    /**
     * Display admin notice after bulk action
     */
    public static function bulk_action_admin_notice() {
        if ( ! empty( $_REQUEST['atoship_synced'] ) || ! empty( $_REQUEST['atoship_sync_failed'] ) ) {
            $synced = isset( $_REQUEST['atoship_synced'] ) ? absint( $_REQUEST['atoship_synced'] ) : 0;
            $failed = isset( $_REQUEST['atoship_sync_failed'] ) ? absint( $_REQUEST['atoship_sync_failed'] ) : 0;

            if ( $synced > 0 ) {
                printf(
                    '<div class="notice notice-success is-dismissible"><p>%s</p></div>',
                    sprintf(
                        /* translators: %d: Number of orders synced */
                        esc_html( _n( '%d order synced to AtoShip.', '%d orders synced to AtoShip.', $synced, 'atoship-for-woocommerce' ) ),
                        $synced
                    )
                );
            }

            if ( $failed > 0 ) {
                printf(
                    '<div class="notice notice-error is-dismissible"><p>%s</p></div>',
                    sprintf(
                        /* translators: %d: Number of orders that failed to sync */
                        esc_html( _n( '%d order failed to sync.', '%d orders failed to sync.', $failed, 'atoship-for-woocommerce' ) ),
                        $failed
                    )
                );
            }
        }
    }

    /**
     * Add order meta box
     */
    public static function add_order_meta_box() {
        $screen = wc_get_container()->get( \Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController::class )->custom_orders_table_usage_is_enabled()
            ? wc_get_page_screen_id( 'shop-order' )
            : 'shop_order';

        add_meta_box(
            'atoship-order-shipping',
            __( 'AtoShip Shipping', 'atoship-for-woocommerce' ),
            array( __CLASS__, 'render_order_meta_box' ),
            $screen,
            'side',
            'high'
        );
    }

    /**
     * Render order meta box
     *
     * @param WP_Post|WC_Order $post_or_order Post object or Order object
     */
    public static function render_order_meta_box( $post_or_order ) {
        $order = $post_or_order instanceof WC_Order ? $post_or_order : wc_get_order( $post_or_order->ID );
        if ( ! $order ) {
            return;
        }

        $atoship_order_id = $order->get_meta( '_atoship_order_id' );
        $tracking_number  = $order->get_meta( '_atoship_tracking_number' );
        $carrier          = $order->get_meta( '_atoship_carrier' );
        $tracking_url     = $order->get_meta( '_atoship_tracking_url' );
        $tracking_status  = $order->get_meta( '_atoship_tracking_status' );
        $label_url        = $order->get_meta( '_atoship_label_url' );
        $sync_error       = $order->get_meta( '_atoship_sync_error' );

        wp_nonce_field( 'atoship_admin', 'atoship_nonce' );
        ?>
        <div class="atoship-meta-box">
            <?php if ( ! empty( $sync_error ) ) : ?>
                <div class="atoship-error">
                    <strong><?php esc_html_e( 'Sync Error:', 'atoship-for-woocommerce' ); ?></strong>
                    <?php echo esc_html( $sync_error ); ?>
                </div>
            <?php endif; ?>

            <?php if ( empty( $atoship_order_id ) ) : ?>
                <p class="atoship-not-synced">
                    <?php esc_html_e( 'Order not synced to AtoShip yet.', 'atoship-for-woocommerce' ); ?>
                </p>
                <button type="button" class="button button-primary atoship-sync-btn" data-order-id="<?php echo esc_attr( $order->get_id() ); ?>">
                    <?php esc_html_e( 'Sync to AtoShip', 'atoship-for-woocommerce' ); ?>
                </button>
            <?php else : ?>
                <table class="atoship-info-table">
                    <tr>
                        <th><?php esc_html_e( 'AtoShip ID:', 'atoship-for-woocommerce' ); ?></th>
                        <td><?php echo esc_html( $atoship_order_id ); ?></td>
                    </tr>
                    <?php if ( ! empty( $carrier ) ) : ?>
                    <tr>
                        <th><?php esc_html_e( 'Carrier:', 'atoship-for-woocommerce' ); ?></th>
                        <td><?php echo esc_html( $carrier ); ?></td>
                    </tr>
                    <?php endif; ?>
                    <?php if ( ! empty( $tracking_number ) ) : ?>
                    <tr>
                        <th><?php esc_html_e( 'Tracking:', 'atoship-for-woocommerce' ); ?></th>
                        <td>
                            <?php if ( ! empty( $tracking_url ) ) : ?>
                                <a href="<?php echo esc_url( $tracking_url ); ?>" target="_blank">
                                    <?php echo esc_html( $tracking_number ); ?>
                                </a>
                            <?php else : ?>
                                <?php echo esc_html( $tracking_number ); ?>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php endif; ?>
                    <?php if ( ! empty( $tracking_status ) ) : ?>
                    <tr>
                        <th><?php esc_html_e( 'Status:', 'atoship-for-woocommerce' ); ?></th>
                        <td>
                            <span class="atoship-tracking-status atoship-status-<?php echo esc_attr( sanitize_html_class( $tracking_status ) ); ?>">
                                <?php echo esc_html( ucfirst( $tracking_status ) ); ?>
                            </span>
                        </td>
                    </tr>
                    <?php endif; ?>
                </table>

                <div class="atoship-actions">
                    <?php if ( ! empty( $label_url ) ) : ?>
                        <a href="<?php echo esc_url( $label_url ); ?>" target="_blank" class="button">
                            <?php esc_html_e( 'View Label', 'atoship-for-woocommerce' ); ?>
                        </a>
                    <?php endif; ?>
                    <a href="<?php echo esc_url( rtrim( ATOSHIP_APP_URL, '/' ) . '/orders/' . $atoship_order_id ); ?>" target="_blank" class="button">
                        <?php esc_html_e( 'View in AtoShip', 'atoship-for-woocommerce' ); ?>
                    </a>
                </div>
            <?php endif; ?>
        </div>

        <style>
            .atoship-meta-box { padding: 5px 0; }
            .atoship-error { background: #f8d7da; color: #721c24; padding: 8px; margin-bottom: 10px; border-radius: 3px; font-size: 12px; }
            .atoship-not-synced { color: #666; margin-bottom: 10px; }
            .atoship-info-table { width: 100%; font-size: 12px; }
            .atoship-info-table th { text-align: left; padding: 4px 0; color: #666; width: 80px; }
            .atoship-info-table td { padding: 4px 0; }
            .atoship-actions { margin-top: 10px; }
            .atoship-actions .button { margin-right: 5px; margin-bottom: 5px; }
            .atoship-tracking-status { display: inline-block; padding: 2px 6px; border-radius: 3px; font-size: 11px; }
            .atoship-status-delivered { background: #d4edda; color: #155724; }
            .atoship-status-in_transit { background: #cce5ff; color: #004085; }
            .atoship-status-exception { background: #f8d7da; color: #721c24; }
        </style>
        <?php
    }

    /**
     * Add column to orders list
     *
     * @param array $columns Existing columns
     * @return array
     */
    public static function add_order_column( $columns ) {
        $new_columns = array();

        foreach ( $columns as $key => $value ) {
            $new_columns[ $key ] = $value;
            if ( 'order_status' === $key ) {
                $new_columns['atoship_status'] = __( 'Shipping', 'atoship-for-woocommerce' );
            }
        }

        return $new_columns;
    }

    /**
     * Render order column content (HPOS)
     *
     * @param string   $column_name Column name
     * @param WC_Order $order       Order object
     */
    public static function render_order_column( $column_name, $order ) {
        if ( 'atoship_status' !== $column_name ) {
            return;
        }

        self::render_shipping_status( $order );
    }

    /**
     * Render order column content (Legacy)
     *
     * @param string $column_name Column name
     * @param int    $order_id    Order ID
     */
    public static function render_order_column_legacy( $column_name, $order_id ) {
        if ( 'atoship_status' !== $column_name ) {
            return;
        }

        $order = wc_get_order( $order_id );
        if ( $order ) {
            self::render_shipping_status( $order );
        }
    }

    /**
     * Render shipping status for order column
     *
     * @param WC_Order $order Order object
     */
    private static function render_shipping_status( $order ) {
        $tracking_number = $order->get_meta( '_atoship_tracking_number' );
        $tracking_status = $order->get_meta( '_atoship_tracking_status' );
        $atoship_id      = $order->get_meta( '_atoship_order_id' );

        if ( ! empty( $tracking_number ) ) {
            $status_class = ! empty( $tracking_status ) ? 'atoship-col-status-' . sanitize_html_class( $tracking_status ) : '';
            printf(
                '<span class="atoship-col-tracking %s" title="%s">%s</span>',
                esc_attr( $status_class ),
                esc_attr( $tracking_status ?: __( 'Shipped', 'atoship-for-woocommerce' ) ),
                esc_html( $tracking_number )
            );
        } elseif ( ! empty( $atoship_id ) ) {
            echo '<span class="atoship-col-synced">' . esc_html__( 'Synced', 'atoship-for-woocommerce' ) . '</span>';
        } else {
            echo '<span class="atoship-col-not-synced">—</span>';
        }
    }

    /**
     * Add single order actions
     *
     * @param array $actions Existing actions
     * @return array
     */
    public static function add_order_actions( $actions ) {
        global $theorder;

        if ( $theorder ) {
            $atoship_order_id = $theorder->get_meta( '_atoship_order_id' );
            if ( empty( $atoship_order_id ) ) {
                $actions['atoship_sync'] = __( 'Sync to AtoShip', 'atoship-for-woocommerce' );
            }
        }

        return $actions;
    }

    /**
     * Process single order action - sync
     *
     * @param WC_Order $order Order object
     */
    public static function process_order_action_sync( $order ) {
        AtoShip_Order_Sync::sync_order( $order );
    }

    /**
     * Enqueue admin scripts
     *
     * @param string $hook Current admin page
     */
    public static function enqueue_scripts( $hook ) {
        $screen = get_current_screen();
        if ( ! $screen ) {
            return;
        }

        // Check if we're on an order edit page or orders list
        $is_order_page = in_array( $screen->id, array( 'shop_order', 'woocommerce_page_wc-orders' ), true ) ||
                         strpos( $screen->id, 'shop_order' ) !== false;

        if ( ! $is_order_page ) {
            return;
        }

        wp_enqueue_script(
            'atoship-orders',
            ATOSHIP_PLUGIN_URL . 'admin/js/atoship-orders.js',
            array( 'jquery' ),
            ATOSHIP_VERSION,
            true
        );

        wp_localize_script( 'atoship-orders', 'atoshipOrders', array(
            'ajaxUrl' => admin_url( 'admin-ajax.php' ),
            'nonce'   => wp_create_nonce( 'atoship_admin' ),
            'i18n'    => array(
                'syncing'  => __( 'Syncing...', 'atoship-for-woocommerce' ),
                'synced'   => __( 'Synced!', 'atoship-for-woocommerce' ),
                'error'    => __( 'Sync failed', 'atoship-for-woocommerce' ),
                'confirm'  => __( 'Sync selected orders to AtoShip?', 'atoship-for-woocommerce' ),
            ),
        ) );

        wp_add_inline_style( 'woocommerce_admin_styles', '
            .atoship-col-tracking { font-family: monospace; font-size: 11px; }
            .atoship-col-synced { color: #0073aa; }
            .atoship-col-not-synced { color: #999; }
            .atoship-col-status-delivered { color: #155724; }
            .atoship-col-status-exception { color: #721c24; }
        ' );
    }
}
