<?php
/**
 * AtoShip Order Sync
 *
 * Handles syncing WooCommerce orders to AtoShip
 *
 * @package AtoShip
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AtoShip_Order_Sync {

    /**
     * Initialize order sync
     */
    public static function init() {
        // Auto sync on order status change
        add_action( 'woocommerce_order_status_changed', array( __CLASS__, 'maybe_sync_order' ), 10, 4 );

        // Manual sync via AJAX
        add_action( 'wp_ajax_atoship_sync_order', array( __CLASS__, 'ajax_sync_order' ) );
        add_action( 'wp_ajax_atoship_sync_orders_batch', array( __CLASS__, 'ajax_sync_orders_batch' ) );
    }

    /**
     * Maybe sync order when status changes
     *
     * @param int      $order_id   Order ID
     * @param string   $old_status Old status
     * @param string   $new_status New status
     * @param WC_Order $order      Order object
     */
    public static function maybe_sync_order( $order_id, $old_status, $new_status, $order ) {
        // Check if auto sync is enabled
        if ( 'yes' !== get_option( 'atoship_auto_sync', 'yes' ) ) {
            return;
        }

        // Check if API is configured
        $api = AtoShip_API::instance();
        if ( ! $api->is_configured() ) {
            return;
        }

        // Get allowed statuses for sync
        $sync_statuses = get_option( 'atoship_sync_statuses', array( 'processing', 'on-hold' ) );
        if ( ! is_array( $sync_statuses ) ) {
            $sync_statuses = array( 'processing', 'on-hold' );
        }

        // Normalize status (remove 'wc-' prefix if present)
        $new_status_normalized = str_replace( 'wc-', '', $new_status );

        // Check if new status is in allowed list
        if ( ! in_array( $new_status_normalized, $sync_statuses, true ) ) {
            return;
        }

        // Check if order was already synced
        $atoship_order_id = $order->get_meta( '_atoship_order_id' );
        if ( ! empty( $atoship_order_id ) ) {
            // Order already synced, could update instead
            return;
        }

        // Sync the order
        self::sync_order( $order );
    }

    /**
     * Sync single order to AtoShip
     *
     * @param WC_Order|int $order Order object or ID
     * @return array|WP_Error
     */
    public static function sync_order( $order ) {
        if ( is_numeric( $order ) ) {
            $order = wc_get_order( $order );
        }

        if ( ! $order instanceof WC_Order ) {
            return new WP_Error( 'invalid_order', __( 'Invalid order.', 'atoship-for-woocommerce' ) );
        }

        $api    = AtoShip_API::instance();
        $result = $api->create_order( $order );

        if ( is_wp_error( $result ) ) {
            // Log the error
            self::log( sprintf(
                'Failed to sync order #%d: %s',
                $order->get_id(),
                $result->get_error_message()
            ), 'error' );

            // Save error to order meta
            $order->update_meta_data( '_atoship_sync_error', $result->get_error_message() );
            $order->update_meta_data( '_atoship_sync_error_time', current_time( 'mysql' ) );
            $order->save();

            return $result;
        }

        // Save AtoShip order ID to WooCommerce order
        if ( isset( $result['id'] ) ) {
            $order->update_meta_data( '_atoship_order_id', $result['id'] );
            $order->update_meta_data( '_atoship_synced_at', current_time( 'mysql' ) );
            $order->delete_meta_data( '_atoship_sync_error' );
            $order->delete_meta_data( '_atoship_sync_error_time' );
            $order->save();

            // Add order note
            $order->add_order_note(
                sprintf(
                    /* translators: %s: AtoShip order ID */
                    __( 'Order synced to AtoShip. AtoShip Order ID: %s', 'atoship-for-woocommerce' ),
                    $result['id']
                ),
                false,
                true
            );

            self::log( sprintf( 'Order #%d synced to AtoShip. AtoShip ID: %s', $order->get_id(), $result['id'] ) );
        }

        return $result;
    }

    /**
     * Sync multiple orders to AtoShip
     *
     * @param array $order_ids Array of order IDs
     * @return array Results with success and failed arrays
     */
    public static function sync_orders_batch( $order_ids ) {
        $results = array(
            'success' => array(),
            'failed'  => array(),
        );

        $orders = array();
        foreach ( $order_ids as $order_id ) {
            $order = wc_get_order( $order_id );
            if ( $order ) {
                // Skip already synced orders
                $atoship_order_id = $order->get_meta( '_atoship_order_id' );
                if ( empty( $atoship_order_id ) ) {
                    $orders[] = $order;
                }
            }
        }

        if ( empty( $orders ) ) {
            return $results;
        }

        // If only one order, use single sync
        if ( count( $orders ) === 1 ) {
            $result = self::sync_order( $orders[0] );
            if ( is_wp_error( $result ) ) {
                $results['failed'][] = array(
                    'order_id' => $orders[0]->get_id(),
                    'error'    => $result->get_error_message(),
                );
            } else {
                $results['success'][] = $orders[0]->get_id();
            }
            return $results;
        }

        // Batch sync
        $api    = AtoShip_API::instance();
        $result = $api->create_orders_batch( $orders );

        if ( is_wp_error( $result ) ) {
            // All failed
            foreach ( $orders as $order ) {
                $results['failed'][] = array(
                    'order_id' => $order->get_id(),
                    'error'    => $result->get_error_message(),
                );
            }
            return $results;
        }

        // Process batch results
        if ( isset( $result['orders'] ) && is_array( $result['orders'] ) ) {
            foreach ( $result['orders'] as $index => $order_result ) {
                $order = isset( $orders[ $index ] ) ? $orders[ $index ] : null;
                if ( ! $order ) {
                    continue;
                }

                if ( isset( $order_result['error'] ) ) {
                    $results['failed'][] = array(
                        'order_id' => $order->get_id(),
                        'error'    => $order_result['error'],
                    );
                    $order->update_meta_data( '_atoship_sync_error', $order_result['error'] );
                    $order->save();
                } else {
                    $results['success'][] = $order->get_id();
                    $order->update_meta_data( '_atoship_order_id', $order_result['id'] );
                    $order->update_meta_data( '_atoship_synced_at', current_time( 'mysql' ) );
                    $order->delete_meta_data( '_atoship_sync_error' );
                    $order->save();

                    $order->add_order_note(
                        sprintf(
                            __( 'Order synced to AtoShip. AtoShip Order ID: %s', 'atoship-for-woocommerce' ),
                            $order_result['id']
                        ),
                        false,
                        true
                    );
                }
            }
        }

        return $results;
    }

    /**
     * AJAX handler for syncing single order
     */
    public static function ajax_sync_order() {
        check_ajax_referer( 'atoship_admin', 'nonce' );

        if ( ! current_user_can( 'edit_shop_orders' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'atoship-for-woocommerce' ) ) );
        }

        $order_id = isset( $_POST['order_id'] ) ? absint( $_POST['order_id'] ) : 0;
        if ( ! $order_id ) {
            wp_send_json_error( array( 'message' => __( 'Invalid order ID.', 'atoship-for-woocommerce' ) ) );
        }

        $result = self::sync_order( $order_id );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( array( 'message' => $result->get_error_message() ) );
        }

        wp_send_json_success( array(
            'message'         => __( 'Order synced successfully!', 'atoship-for-woocommerce' ),
            'atoship_order_id' => $result['id'],
        ) );
    }

    /**
     * AJAX handler for batch syncing orders
     */
    public static function ajax_sync_orders_batch() {
        check_ajax_referer( 'atoship_admin', 'nonce' );

        if ( ! current_user_can( 'edit_shop_orders' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'atoship-for-woocommerce' ) ) );
        }

        $order_ids = isset( $_POST['order_ids'] ) ? array_map( 'absint', (array) $_POST['order_ids'] ) : array();
        if ( empty( $order_ids ) ) {
            wp_send_json_error( array( 'message' => __( 'No orders selected.', 'atoship-for-woocommerce' ) ) );
        }

        $results = self::sync_orders_batch( $order_ids );

        wp_send_json_success( array(
            'message' => sprintf(
                /* translators: 1: Number of successful syncs, 2: Number of failed syncs */
                __( 'Synced %1$d orders. %2$d failed.', 'atoship-for-woocommerce' ),
                count( $results['success'] ),
                count( $results['failed'] )
            ),
            'results' => $results,
        ) );
    }

    /**
     * Log message
     *
     * @param string $message Message to log
     * @param string $level   Log level
     */
    private static function log( $message, $level = 'info' ) {
        if ( function_exists( 'wc_get_logger' ) ) {
            $logger = wc_get_logger();
            $logger->log( $level, $message, array( 'source' => 'atoship' ) );
        }
    }
}
