<?php
/**
 * AtoShip Settings
 *
 * Handles plugin settings page and WooCommerce integration settings
 *
 * @package AtoShip
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AtoShip_Settings {

    /**
     * Initialize settings
     */
    public static function init() {
        add_filter( 'woocommerce_settings_tabs_array', array( __CLASS__, 'add_settings_tab' ), 50 );
        add_action( 'woocommerce_settings_tabs_atoship', array( __CLASS__, 'settings_tab' ) );
        add_action( 'woocommerce_update_options_atoship', array( __CLASS__, 'update_settings' ) );
        add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_admin_scripts' ) );
        add_action( 'wp_ajax_atoship_test_connection', array( __CLASS__, 'ajax_test_connection' ) );
        add_action( 'admin_notices', array( __CLASS__, 'admin_notices' ) );
    }

    /**
     * Add settings tab to WooCommerce settings
     *
     * @param array $tabs Existing tabs
     * @return array
     */
    public static function add_settings_tab( $tabs ) {
        $tabs['atoship'] = __( 'AtoShip', 'atoship-for-woocommerce' );
        return $tabs;
    }

    /**
     * Render settings tab content
     */
    public static function settings_tab() {
        // Show OAuth success/error messages
        self::render_oauth_notices();

        woocommerce_admin_fields( self::get_settings() );
        self::render_oauth_connection();
        self::render_connection_status();
        self::render_webhook_info();
    }

    /**
     * Update settings
     */
    public static function update_settings() {
        woocommerce_update_options( self::get_settings() );

        // Clear API instance cache when settings are updated
        // This ensures the new API key is used
        delete_transient( 'atoship_connection_status' );
    }

    /**
     * Get settings fields
     *
     * @return array
     */
    public static function get_settings() {
        $settings = array(
            // Section: Sync Settings
            array(
                'title' => __( 'Order Sync Settings', 'atoship-for-woocommerce' ),
                'type'  => 'title',
                'desc'  => __( 'Configure how orders are synced to AtoShip.', 'atoship-for-woocommerce' ),
                'id'    => 'atoship_sync_settings',
            ),
            array(
                'title'   => __( 'Auto Sync Orders', 'atoship-for-woocommerce' ),
                'desc'    => __( 'Automatically sync new orders to AtoShip', 'atoship-for-woocommerce' ),
                'id'      => 'atoship_auto_sync',
                'type'    => 'checkbox',
                'default' => 'yes',
            ),
            array(
                'title'    => __( 'Sync Order Statuses', 'atoship-for-woocommerce' ),
                'desc'     => __( 'Only sync orders with these statuses', 'atoship-for-woocommerce' ),
                'id'       => 'atoship_sync_statuses',
                'type'     => 'multiselect',
                'class'    => 'wc-enhanced-select',
                'css'      => 'min-width: 350px;',
                'default'  => array( 'processing', 'on-hold' ),
                'options'  => wc_get_order_statuses(),
                'desc_tip' => true,
            ),
            array(
                'title'   => __( 'Sync Completed Orders', 'atoship-for-woocommerce' ),
                'desc'    => __( 'Also sync orders that are already marked as completed', 'atoship-for-woocommerce' ),
                'id'      => 'atoship_sync_completed',
                'type'    => 'checkbox',
                'default' => 'no',
            ),
            array(
                'type' => 'sectionend',
                'id'   => 'atoship_sync_settings',
            ),

            // Section: Tracking Settings
            array(
                'title' => __( 'Tracking Settings', 'atoship-for-woocommerce' ),
                'type'  => 'title',
                'desc'  => __( 'Configure tracking number behavior.', 'atoship-for-woocommerce' ),
                'id'    => 'atoship_tracking_settings',
            ),
            array(
                'title'   => __( 'Auto Complete Orders', 'atoship-for-woocommerce' ),
                'desc'    => __( 'Automatically mark orders as completed when tracking shows delivered', 'atoship-for-woocommerce' ),
                'id'      => 'atoship_auto_complete',
                'type'    => 'checkbox',
                'default' => 'no',
            ),
            array(
                'title'   => __( 'Send Tracking Email', 'atoship-for-woocommerce' ),
                'desc'    => __( 'Send tracking information email to customers when shipment is created', 'atoship-for-woocommerce' ),
                'id'      => 'atoship_send_tracking_email',
                'type'    => 'checkbox',
                'default' => 'yes',
            ),
            array(
                'type' => 'sectionend',
                'id'   => 'atoship_tracking_settings',
            ),

            // Section: Advanced Settings
            array(
                'title' => __( 'Advanced Settings', 'atoship-for-woocommerce' ),
                'type'  => 'title',
                'id'    => 'atoship_advanced_settings',
            ),
            array(
                'title'   => __( 'Debug Log', 'atoship-for-woocommerce' ),
                'desc'    => __( 'Enable logging for debugging (logs are saved in WooCommerce > Status > Logs)', 'atoship-for-woocommerce' ),
                'id'      => 'atoship_debug_log',
                'type'    => 'checkbox',
                'default' => 'no',
            ),
            array(
                'type' => 'sectionend',
                'id'   => 'atoship_advanced_settings',
            ),
        );

        return apply_filters( 'atoship_settings', $settings );
    }

    /**
     * Render OAuth notices
     */
    private static function render_oauth_notices() {
        $success = get_transient( 'atoship_oauth_success' );
        $error   = get_transient( 'atoship_oauth_error' );

        if ( $success ) {
            delete_transient( 'atoship_oauth_success' );
            ?>
            <div class="notice notice-success is-dismissible">
                <p><?php echo esc_html( $success ); ?></p>
            </div>
            <?php
        }

        if ( $error ) {
            delete_transient( 'atoship_oauth_error' );
            ?>
            <div class="notice notice-error is-dismissible">
                <p><?php echo esc_html( $error ); ?></p>
            </div>
            <?php
        }
    }

    /**
     * Render OAuth connection section
     */
    private static function render_oauth_connection() {
        $is_connected = AtoShip_OAuth::is_connected();
        $account_info = $is_connected ? AtoShip_OAuth::get_account_info() : false;
        ?>
        <h2><?php esc_html_e( 'AtoShip Connection', 'atoship-for-woocommerce' ); ?></h2>
        <p class="description">
            <?php esc_html_e( 'Connect your store to AtoShip with one click to start creating discounted shipping labels.', 'atoship-for-woocommerce' ); ?>
        </p>
        <table class="form-table">
            <tr>
                <th scope="row"><?php esc_html_e( 'OAuth Status', 'atoship-for-woocommerce' ); ?></th>
                <td>
                    <?php if ( $is_connected ) : ?>
                        <span class="atoship-status atoship-status-connected">
                            ✓ <?php esc_html_e( 'Connected', 'atoship-for-woocommerce' ); ?>
                        </span>
                        <?php if ( $account_info ) : ?>
                            <p class="description">
                                <?php
                                printf(
                                    /* translators: %s: Connected account email */
                                    esc_html__( 'Connected as: %s', 'atoship-for-woocommerce' ),
                                    '<strong>' . esc_html( $account_info['email'] ?? '' ) . '</strong>'
                                );
                                ?>
                            </p>
                        <?php endif; ?>
                        <p style="margin-top: 10px;">
                            <button type="button" class="button" id="atoship-disconnect-oauth">
                                <?php esc_html_e( 'Disconnect OAuth', 'atoship-for-woocommerce' ); ?>
                            </button>
                        </p>
                    <?php else : ?>
                        <span class="atoship-status atoship-status-not-configured">
                            <?php esc_html_e( 'Not Connected', 'atoship-for-woocommerce' ); ?>
                        </span>
                        <p style="margin-top: 10px;">
                            <a href="<?php echo esc_url( AtoShip_OAuth::get_authorize_url() ); ?>" class="button button-primary">
                                <?php esc_html_e( 'Connect with AtoShip', 'atoship-for-woocommerce' ); ?>
                            </a>
                        </p>
                    <?php endif; ?>
                </td>
            </tr>
        </table>
        <?php
    }

    /**
     * Render connection status section
     */
    private static function render_connection_status() {
        $api = AtoShip_API::instance();
        ?>
        <h2><?php esc_html_e( 'Connection Status', 'atoship-for-woocommerce' ); ?></h2>
        <table class="form-table">
            <tr>
                <th scope="row"><?php esc_html_e( 'Status', 'atoship-for-woocommerce' ); ?></th>
                <td>
                    <span id="atoship-connection-status">
                        <?php if ( ! $api->is_configured() ) : ?>
                            <span class="atoship-status atoship-status-not-configured">
                                <?php esc_html_e( 'Not Configured', 'atoship-for-woocommerce' ); ?>
                            </span>
                        <?php else : ?>
                            <span class="atoship-status atoship-status-unknown">
                                <?php esc_html_e( 'Click "Test Connection" to verify', 'atoship-for-woocommerce' ); ?>
                            </span>
                        <?php endif; ?>
                    </span>
                    <?php if ( $api->is_configured() ) : ?>
                        <button type="button" class="button" id="atoship-test-connection">
                            <?php esc_html_e( 'Test Connection', 'atoship-for-woocommerce' ); ?>
                        </button>
                        <span class="spinner" style="float: none; margin-top: 0;"></span>
                    <?php endif; ?>
                </td>
            </tr>
        </table>
        <?php
    }

    /**
     * Render webhook information section
     */
    private static function render_webhook_info() {
        $webhook_url    = rest_url( 'atoship/v1/webhook' );
        $webhook_secret = get_option( 'atoship_webhook_secret', '' );
        ?>
        <h2><?php esc_html_e( 'Webhook Configuration', 'atoship-for-woocommerce' ); ?></h2>
        <p class="description">
            <?php esc_html_e( 'Configure these settings in your AtoShip dashboard to receive real-time updates.', 'atoship-for-woocommerce' ); ?>
        </p>
        <table class="form-table">
            <tr>
                <th scope="row"><?php esc_html_e( 'Webhook URL', 'atoship-for-woocommerce' ); ?></th>
                <td>
                    <code id="atoship-webhook-url"><?php echo esc_url( $webhook_url ); ?></code>
                    <button type="button" class="button button-small atoship-copy-btn" data-copy="atoship-webhook-url">
                        <?php esc_html_e( 'Copy', 'atoship-for-woocommerce' ); ?>
                    </button>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e( 'Webhook Secret', 'atoship-for-woocommerce' ); ?></th>
                <td>
                    <code id="atoship-webhook-secret"><?php echo esc_html( $webhook_secret ); ?></code>
                    <button type="button" class="button button-small atoship-copy-btn" data-copy="atoship-webhook-secret">
                        <?php esc_html_e( 'Copy', 'atoship-for-woocommerce' ); ?>
                    </button>
                    <button type="button" class="button button-small" id="atoship-regenerate-secret">
                        <?php esc_html_e( 'Regenerate', 'atoship-for-woocommerce' ); ?>
                    </button>
                </td>
            </tr>
        </table>
        <style>
            .atoship-status, .atoship-mode {
                display: inline-block;
                padding: 3px 8px;
                border-radius: 3px;
                font-size: 12px;
                font-weight: 600;
            }
            .atoship-status-connected { background: #d4edda; color: #155724; }
            .atoship-status-error { background: #f8d7da; color: #721c24; }
            .atoship-status-not-configured { background: #fff3cd; color: #856404; }
            .atoship-status-unknown { background: #e2e3e5; color: #383d41; }
            .atoship-mode-test { background: #fff3cd; color: #856404; }
            .atoship-mode-live { background: #d4edda; color: #155724; }
        </style>
        <?php
    }

    /**
     * Enqueue admin scripts
     *
     * @param string $hook Current admin page
     */
    public static function enqueue_admin_scripts( $hook ) {
        if ( 'woocommerce_page_wc-settings' !== $hook ) {
            return;
        }

        if ( ! isset( $_GET['tab'] ) || 'atoship' !== $_GET['tab'] ) {
            return;
        }

        wp_enqueue_script(
            'atoship-admin',
            ATOSHIP_PLUGIN_URL . 'admin/js/atoship-admin.js',
            array( 'jquery' ),
            ATOSHIP_VERSION,
            true
        );

        wp_localize_script( 'atoship-admin', 'atoshipAdmin', array(
            'ajaxUrl' => admin_url( 'admin-ajax.php' ),
            'nonce'   => wp_create_nonce( 'atoship_admin' ),
            'i18n'    => array(
                'testing'    => __( 'Testing...', 'atoship-for-woocommerce' ),
                'connected'  => __( 'Connected', 'atoship-for-woocommerce' ),
                'error'      => __( 'Connection Failed', 'atoship-for-woocommerce' ),
                'copied'     => __( 'Copied!', 'atoship-for-woocommerce' ),
                'copy'       => __( 'Copy', 'atoship-for-woocommerce' ),
            ),
        ) );
    }

    /**
     * AJAX handler for testing connection
     */
    public static function ajax_test_connection() {
        check_ajax_referer( 'atoship_admin', 'nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'atoship-for-woocommerce' ) ) );
        }

        $api    = AtoShip_API::instance();
        $result = $api->test_connection();

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( array(
                'message' => $result->get_error_message(),
            ) );
        }

        wp_send_json_success( array(
            'message' => __( 'Successfully connected to AtoShip!', 'atoship-for-woocommerce' ),
            'account' => $result,
        ) );
    }

    /**
     * Admin notices
     */
    public static function admin_notices() {
        // Only show on WooCommerce pages
        $screen = get_current_screen();
        if ( ! $screen || strpos( $screen->id, 'woocommerce' ) === false ) {
            return;
        }

        $api = AtoShip_API::instance();

        // Check if API key is configured
        if ( ! $api->is_configured() ) {
            ?>
            <div class="notice notice-warning">
                <p>
                    <?php
                    printf(
                        /* translators: %s: Settings page URL */
                        esc_html__( 'AtoShip for WooCommerce is not configured. Please %s to start shipping.', 'atoship-for-woocommerce' ),
                        '<a href="' . esc_url( admin_url( 'admin.php?page=wc-settings&tab=atoship' ) ) . '">' . esc_html__( 'connect your store', 'atoship-for-woocommerce' ) . '</a>'
                    );
                    ?>
                </p>
            </div>
            <?php
        }
    }
}
