<?php
/**
 * AtoShip Shipping Method
 *
 * WooCommerce shipping method that provides real-time rates from AtoShip API
 * at checkout. Extends WC_Shipping_Method for full WooCommerce integration.
 *
 * @package AtoShip
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Register the shipping method with WooCommerce
 */
function atoship_shipping_method_init() {
    if ( ! class_exists( 'WC_Shipping_Method' ) ) {
        return;
    }

    /**
     * WC_AtoShip_Shipping_Method
     *
     * Provides real-time shipping rates from AtoShip's multi-carrier API.
     * Supports USPS, FedEx, UPS, DHL and 180+ carriers through a single integration.
     */
    class WC_AtoShip_Shipping_Method extends WC_Shipping_Method {

        /**
         * Rate cache TTL in seconds
         */
        const CACHE_TTL = 300; // 5 minutes

        /**
         * Constructor
         *
         * @param int $instance_id Shipping zone instance ID
         */
        public function __construct( $instance_id = 0 ) {
            $this->id                 = 'atoship';
            $this->instance_id        = absint( $instance_id );
            $this->method_title       = __( 'AtoShip Shipping', 'atoship-for-woocommerce' );
            $this->method_description = __( 'Real-time shipping rates from USPS, FedEx, UPS, and more via AtoShip.', 'atoship-for-woocommerce' );
            $this->supports           = array(
                'shipping-zones',
                'instance-settings',
                'instance-settings-modal',
            );

            $this->init();
        }

        /**
         * Initialize settings
         */
        public function init() {
            $this->init_instance_form_fields();
            $this->init_settings();

            $this->title      = $this->get_option( 'title', __( 'AtoShip Shipping', 'atoship-for-woocommerce' ) );
            $this->enabled    = $this->get_option( 'enabled', 'yes' );
            $this->tax_status = $this->get_option( 'tax_status', 'none' );

            add_action( 'woocommerce_update_options_shipping_' . $this->id, array( $this, 'process_admin_options' ) );
        }

        /**
         * Instance settings fields (per shipping zone)
         */
        public function init_instance_form_fields() {
            $this->instance_form_fields = array(
                'title' => array(
                    'title'   => __( 'Method Title', 'atoship-for-woocommerce' ),
                    'type'    => 'text',
                    'default' => __( 'AtoShip Shipping', 'atoship-for-woocommerce' ),
                    'desc_tip' => true,
                    'description' => __( 'This is the title displayed to customers during checkout.', 'atoship-for-woocommerce' ),
                ),
                'tax_status' => array(
                    'title'   => __( 'Tax Status', 'atoship-for-woocommerce' ),
                    'type'    => 'select',
                    'default' => 'none',
                    'options' => array(
                        'taxable' => __( 'Taxable', 'atoship-for-woocommerce' ),
                        'none'    => __( 'Not taxable', 'atoship-for-woocommerce' ),
                    ),
                ),
                'fallback_cost' => array(
                    'title'       => __( 'Fallback Cost', 'atoship-for-woocommerce' ),
                    'type'        => 'text',
                    'default'     => '',
                    'placeholder' => __( 'Leave blank to hide shipping if API fails', 'atoship-for-woocommerce' ),
                    'desc_tip'    => true,
                    'description' => __( 'If the AtoShip API is unavailable, show this flat rate instead. Leave blank to show no shipping option.', 'atoship-for-woocommerce' ),
                ),
                'rate_markup' => array(
                    'title'       => __( 'Rate Markup (%)', 'atoship-for-woocommerce' ),
                    'type'        => 'number',
                    'default'     => '0',
                    'desc_tip'    => true,
                    'description' => __( 'Add a percentage markup to all rates. Use 0 for no markup.', 'atoship-for-woocommerce' ),
                    'custom_attributes' => array(
                        'min'  => '0',
                        'max'  => '100',
                        'step' => '1',
                    ),
                ),
                'rate_adjustment' => array(
                    'title'       => __( 'Rate Adjustment ($)', 'atoship-for-woocommerce' ),
                    'type'        => 'text',
                    'default'     => '0',
                    'desc_tip'    => true,
                    'description' => __( 'Add a fixed amount to all rates. Use negative values for discounts (e.g., -2.00).', 'atoship-for-woocommerce' ),
                ),
                'max_rates' => array(
                    'title'       => __( 'Max Rates Displayed', 'atoship-for-woocommerce' ),
                    'type'        => 'number',
                    'default'     => '5',
                    'desc_tip'    => true,
                    'description' => __( 'Maximum number of shipping options to show at checkout. 0 for all.', 'atoship-for-woocommerce' ),
                    'custom_attributes' => array(
                        'min'  => '0',
                        'max'  => '20',
                        'step' => '1',
                    ),
                ),
                'show_delivery_estimate' => array(
                    'title'   => __( 'Show Delivery Estimate', 'atoship-for-woocommerce' ),
                    'type'    => 'checkbox',
                    'default' => 'yes',
                    'label'   => __( 'Show estimated delivery days next to each rate', 'atoship-for-woocommerce' ),
                ),
            );
        }

        /**
         * Check if this method is available
         *
         * @param array $package Shipping package
         * @return bool
         */
        public function is_available( $package ) {
            $this->log( 'is_available() called. enabled=' . $this->enabled, 'debug' );

            if ( 'no' === $this->enabled ) {
                $this->log( 'is_available() returning false: method disabled', 'debug' );
                return false;
            }

            // Check API is configured
            $api = AtoShip_API::instance();
            $configured = $api->is_configured();
            $this->log( 'is_available() API configured=' . ( $configured ? 'yes' : 'no' ), 'debug' );
            if ( ! $configured ) {
                return false;
            }

            $result = parent::is_available( $package );
            $this->log( 'is_available() parent result=' . ( $result ? 'yes' : 'no' ), 'debug' );
            return $result;
        }

        /**
         * Calculate shipping rates
         *
         * Called by WooCommerce during cart/checkout to get available shipping options.
         *
         * @param array $package WooCommerce shipping package containing destination and cart items
         */
        public function calculate_shipping( $package = array() ) {
            $this->log( 'calculate_shipping() called. Destination: ' . wp_json_encode( $package['destination'] ?? 'empty' ), 'debug' );
            $destination = $package['destination'];

            // Need at minimum a postcode to calculate rates
            if ( empty( $destination['postcode'] ) ) {
                return;
            }

            // Build parcel data from cart items
            $parcel = $this->build_parcel_from_package( $package );
            if ( ! $parcel ) {
                $this->add_fallback_rate();
                return;
            }

            // Build from address from WooCommerce store settings
            $from_address = $this->get_store_from_address();
            if ( ! $from_address ) {
                $this->log( 'Store address not configured. Cannot calculate shipping rates.', 'error' );
                $this->add_fallback_rate();
                return;
            }

            // Build to address from package destination
            $to_address = array(
                'name'    => '',
                'street1' => $destination['address'] ?? '',
                'street2' => $destination['address_2'] ?? '',
                'city'    => $destination['city'] ?? '',
                'state'   => $destination['state'] ?? '',
                'zip'     => $destination['postcode'] ?? '',
                'country' => $destination['country'] ?? 'US',
            );

            // Check cache first
            $cache_key = $this->get_cache_key( $from_address, $to_address, $parcel );
            $cached_rates = get_transient( $cache_key );

            if ( false !== $cached_rates && is_array( $cached_rates ) ) {
                $this->log( 'Using cached rates (' . count( $cached_rates ) . ' rates)', 'debug' );
                foreach ( $cached_rates as $rate ) {
                    $this->add_rate( $rate );
                }
                return;
            }

            // Fetch rates from AtoShip API
            $rates = $this->fetch_rates( $from_address, $to_address, $parcel );

            if ( empty( $rates ) ) {
                $this->add_fallback_rate();
                return;
            }

            // Cache and add rates
            set_transient( $cache_key, $rates, self::CACHE_TTL );

            foreach ( $rates as $rate ) {
                $this->add_rate( $rate );
            }
        }

        /**
         * Fetch rates from AtoShip API
         *
         * @param array $from_address Origin address
         * @param array $to_address   Destination address
         * @param array $parcel       Package dimensions/weight
         * @return array WooCommerce rate arrays
         */
        private function fetch_rates( $from_address, $to_address, $parcel ) {
            $api = AtoShip_API::instance();

            // Build request for AtoShip v1/rates API
            $request_data = array(
                'from_address' => $from_address,
                'to_address'   => $to_address,
                'parcel'       => $parcel,
                'skip_address_validation' => true, // Skip for checkout speed
            );

            $this->log( 'Fetching rates: ' . wp_json_encode( array(
                'from_zip' => $from_address['zip'],
                'to_zip'   => $to_address['zip'],
                'weight'   => $parcel['weight'],
                'weight_unit' => $parcel['weight_unit'],
            ) ), 'debug' );

            $response = $api->get_rates( $request_data );

            if ( is_wp_error( $response ) ) {
                $this->log( 'API error: ' . $response->get_error_message(), 'error' );
                return array();
            }

            // Handle address validation response (should not happen with skip_address_validation)
            if ( isset( $response['object'] ) && 'AddressValidation' === $response['object'] ) {
                $this->log( 'Address validation required, skipping', 'debug' );
                return array();
            }

            if ( ! isset( $response['data'] ) || ! is_array( $response['data'] ) ) {
                $this->log( 'Invalid API response format', 'error' );
                return array();
            }

            $this->log( 'Received ' . count( $response['data'] ) . ' rates from API', 'debug' );

            // Settings
            $markup_percent   = floatval( $this->get_option( 'rate_markup', 0 ) );
            $rate_adjustment  = floatval( $this->get_option( 'rate_adjustment', 0 ) );
            $max_rates        = intval( $this->get_option( 'max_rates', 5 ) );
            $show_estimate    = 'yes' === $this->get_option( 'show_delivery_estimate', 'yes' );

            $wc_rates = array();

            foreach ( $response['data'] as $rate ) {
                $carrier  = $rate['carrier'] ?? 'Unknown';
                $service  = $rate['service'] ?? 'Standard';
                $cost     = floatval( $rate['rate'] ?? 0 );
                $days     = isset( $rate['delivery_days'] ) ? intval( $rate['delivery_days'] ) : null;
                $rate_id  = $rate['id'] ?? '';

                if ( $cost <= 0 ) {
                    continue;
                }

                // Apply markup
                if ( $markup_percent > 0 ) {
                    $cost = $cost * ( 1 + $markup_percent / 100 );
                }

                // Apply fixed adjustment
                if ( $rate_adjustment != 0 ) {
                    $cost = max( 0, $cost + $rate_adjustment );
                }

                $cost = round( $cost, 2 );

                // Build label: "USPS Priority Mail" or "USPS Priority Mail (2-3 days)"
                $label = $carrier . ' ' . $service;
                if ( $show_estimate && $days ) {
                    $label .= ' (' . sprintf(
                        /* translators: %d: number of business days */
                        _n( '%d business day', '%d business days', $days, 'atoship-for-woocommerce' ),
                        $days
                    ) . ')';
                }

                $wc_rates[] = array(
                    'id'        => $this->get_rate_id( $carrier . '_' . sanitize_title( $service ) ),
                    'label'     => $label,
                    'cost'      => $cost,
                    'calc_tax'  => 'per_order',
                    'meta_data' => array(
                        'atoship_rate_id'    => $rate_id,
                        'atoship_carrier'    => $carrier,
                        'atoship_service'    => $service,
                        'atoship_service_code' => $rate['service_code'] ?? '',
                        'atoship_delivery_days' => $days,
                        'atoship_base_rate'  => floatval( $rate['rate'] ?? 0 ),
                    ),
                );
            }

            // Limit number of rates if configured
            if ( $max_rates > 0 && count( $wc_rates ) > $max_rates ) {
                $wc_rates = array_slice( $wc_rates, 0, $max_rates );
            }

            return $wc_rates;
        }

        /**
         * Build parcel object from WooCommerce package
         *
         * Aggregates all cart items into a single parcel with total weight
         * and max dimensions (largest item bounding box).
         *
         * @param array $package WooCommerce shipping package
         * @return array|false Parcel data or false on failure
         */
        private function build_parcel_from_package( $package ) {
            $total_weight = 0;
            $max_length   = 0;
            $max_width    = 0;
            $max_height   = 0;
            $has_dimensions = false;

            $wc_weight_unit = get_option( 'woocommerce_weight_unit', 'kg' );
            $wc_dim_unit    = get_option( 'woocommerce_dimension_unit', 'cm' );

            foreach ( $package['contents'] as $item ) {
                $product = $item['data'];
                $qty     = $item['quantity'];

                if ( ! $product instanceof WC_Product ) {
                    continue;
                }

                // Weight
                $weight = (float) $product->get_weight();
                if ( $weight > 0 ) {
                    $total_weight += $weight * $qty;
                }

                // Dimensions - use largest item's dimensions
                $length = (float) $product->get_length();
                $width  = (float) $product->get_width();
                $height = (float) $product->get_height();

                if ( $length > 0 && $width > 0 && $height > 0 ) {
                    $has_dimensions = true;
                    // For multiple items, use the largest dimensions
                    $max_length = max( $max_length, $length );
                    $max_width  = max( $max_width, $width );
                    // Stack height for multiple quantities of same item
                    $max_height = max( $max_height, $height * $qty );
                }
            }

            // Use default weight if products don't have weight set
            if ( $total_weight <= 0 ) {
                // Default to 1 lb per item when no weight is configured
                $default_weight_per_item = ( 'oz' === $wc_weight_unit ) ? 16 : 1; // 1 lb
                if ( 'kg' === $wc_weight_unit ) {
                    $default_weight_per_item = 0.45; // ~1 lb in kg
                } elseif ( 'g' === $wc_weight_unit ) {
                    $default_weight_per_item = 450; // ~1 lb in g
                }
                $total_qty = array_sum( array_map( function( $item ) {
                    return $item['quantity'];
                }, $package['contents'] ) );
                $total_weight = $default_weight_per_item * max( $total_qty, 1 );
                $this->log( 'Products have no weight - using default ' . $total_weight . ' ' . $wc_weight_unit, 'info' );
            }

            // Convert weight to oz for AtoShip API
            $weight_in_oz = $this->convert_weight_to_oz( $total_weight, $wc_weight_unit );

            // Convert dimensions to inches
            if ( $has_dimensions ) {
                $length_in = $this->convert_dimension_to_in( $max_length, $wc_dim_unit );
                $width_in  = $this->convert_dimension_to_in( $max_width, $wc_dim_unit );
                $height_in = $this->convert_dimension_to_in( $max_height, $wc_dim_unit );
            } else {
                // Default small package dimensions if products don't have dimensions set
                $length_in = 6;
                $width_in  = 4;
                $height_in = 2;
            }

            return array(
                'weight'         => round( $weight_in_oz, 1 ),
                'weight_unit'    => 'oz',
                'length'         => round( $length_in, 1 ),
                'width'          => round( $width_in, 1 ),
                'height'         => round( $height_in, 1 ),
                'dimension_unit' => 'in',
            );
        }

        /**
         * Get store from-address based on WooCommerce settings
         *
         * @return array|false Address array or false if not configured
         */
        private function get_store_from_address() {
            $country_state = get_option( 'woocommerce_default_country', '' );
            $parts = explode( ':', $country_state );
            $country = $parts[0] ?? '';
            $state   = $parts[1] ?? '';
            $city    = get_option( 'woocommerce_store_city', '' );
            $zip     = get_option( 'woocommerce_store_postcode', '' );
            $address = get_option( 'woocommerce_store_address', '' );
            $address_2 = get_option( 'woocommerce_store_address_2', '' );

            if ( empty( $zip ) || empty( $state ) ) {
                return false;
            }

            return array(
                'name'    => get_bloginfo( 'name' ),
                'street1' => $address,
                'street2' => $address_2,
                'city'    => $city,
                'state'   => $state,
                'zip'     => $zip,
                'country' => $country ?: 'US',
            );
        }

        /**
         * Add fallback rate if API fails and fallback is configured
         */
        private function add_fallback_rate() {
            $fallback = $this->get_option( 'fallback_cost', '' );
            if ( '' !== $fallback && is_numeric( $fallback ) ) {
                $this->add_rate( array(
                    'id'       => $this->get_rate_id( 'fallback' ),
                    'label'    => __( 'Standard Shipping', 'atoship-for-woocommerce' ),
                    'cost'     => floatval( $fallback ),
                    'calc_tax' => 'per_order',
                ) );
            }
        }

        /**
         * Generate cache key for rate requests
         *
         * @param array $from  Origin address
         * @param array $to    Destination address
         * @param array $parcel Package data
         * @return string Transient key
         */
        private function get_cache_key( $from, $to, $parcel ) {
            $key_data = array(
                'from_zip'  => $from['zip'] ?? '',
                'from_state' => $from['state'] ?? '',
                'to_zip'    => $to['zip'] ?? '',
                'to_state'  => $to['state'] ?? '',
                'to_country' => $to['country'] ?? 'US',
                'weight'    => $parcel['weight'] ?? 0,
                'length'    => $parcel['length'] ?? 0,
                'width'     => $parcel['width'] ?? 0,
                'height'    => $parcel['height'] ?? 0,
                'inst'      => $this->instance_id,
            );
            return 'atoship_rates_' . md5( wp_json_encode( $key_data ) );
        }

        /**
         * Convert weight to ounces
         *
         * @param float  $weight Weight value
         * @param string $unit   Source unit (kg, g, lbs, oz)
         * @return float Weight in ounces
         */
        private function convert_weight_to_oz( $weight, $unit ) {
            switch ( strtolower( $unit ) ) {
                case 'kg':
                    return $weight * 35.274;
                case 'g':
                    return $weight * 0.035274;
                case 'lbs':
                    return $weight * 16;
                case 'oz':
                default:
                    return $weight;
            }
        }

        /**
         * Convert dimension to inches
         *
         * @param float  $dim  Dimension value
         * @param string $unit Source unit (cm, m, mm, in, yd)
         * @return float Dimension in inches
         */
        private function convert_dimension_to_in( $dim, $unit ) {
            switch ( strtolower( $unit ) ) {
                case 'cm':
                    return $dim * 0.3937;
                case 'm':
                    return $dim * 39.3701;
                case 'mm':
                    return $dim * 0.03937;
                case 'yd':
                    return $dim * 36;
                case 'in':
                default:
                    return $dim;
            }
        }

        /**
         * Log message using WooCommerce logger
         *
         * @param string $message Log message
         * @param string $level   Log level
         */
        private function log( $message, $level = 'info' ) {
            if ( function_exists( 'wc_get_logger' ) && ( 'yes' === get_option( 'atoship_debug_log', 'no' ) || in_array( $level, array( 'error', 'warning' ), true ) ) ) {
                $logger = wc_get_logger();
                $logger->log( $level, '[Shipping Method] ' . $message, array( 'source' => 'atoship' ) );
            }
        }
    }
}
add_action( 'woocommerce_shipping_init', 'atoship_shipping_method_init' );

/**
 * Register the shipping method
 *
 * @param array $methods Existing shipping methods
 * @return array Modified shipping methods
 */
function atoship_add_shipping_method( $methods ) {
    $methods['atoship'] = 'WC_AtoShip_Shipping_Method';
    return $methods;
}
add_filter( 'woocommerce_shipping_methods', 'atoship_add_shipping_method' );

/**
 * Save selected AtoShip rate metadata to order when order is placed
 *
 * This stores the AtoShip rate_id, carrier, and service on the order
 * so we can use them later for label purchase.
 */
function atoship_save_shipping_meta_to_order( $order, $data ) {
    if ( ! $order instanceof WC_Order ) {
        return;
    }

    foreach ( $order->get_shipping_methods() as $shipping_method ) {
        $meta = $shipping_method->get_meta_data();
        foreach ( $meta as $meta_item ) {
            $meta_data = $meta_item->get_data();
            $key = $meta_data['key'] ?? '';
            $value = $meta_data['value'] ?? '';

            if ( strpos( $key, 'atoship_' ) === 0 ) {
                $order->update_meta_data( '_' . $key, $value );
            }
        }
    }

    $order->save();
}
add_action( 'woocommerce_checkout_create_order', 'atoship_save_shipping_meta_to_order', 10, 2 );

/**
 * Show admin notice when products are missing weight
 *
 * Checks published products and alerts the store owner so they can
 * set weights for accurate shipping rate calculation.
 */
function atoship_check_products_missing_weight() {
    // Only show on relevant admin pages
    $screen = get_current_screen();
    if ( ! $screen ) {
        return;
    }

    $relevant_screens = array(
        'edit-product',         // Products list
        'product',              // Product edit
        'woocommerce_page_wc-settings', // WC Settings
        'shop_order',           // Order edit (legacy)
        'woocommerce_page_wc-orders',   // Order edit (HPOS)
    );

    if ( ! in_array( $screen->id, $relevant_screens, true ) ) {
        return;
    }

    // Use transient to avoid querying on every page load
    $products_without_weight = get_transient( 'atoship_products_missing_weight' );

    if ( false === $products_without_weight ) {
        // Query for published simple/variable products with no weight
        $args = array(
            'status'  => 'publish',
            'limit'   => -1,
            'return'  => 'ids',
            'type'    => array( 'simple', 'variable' ),
        );

        $products = wc_get_products( $args );
        $missing  = array();

        foreach ( $products as $product_id ) {
            $product = wc_get_product( $product_id );
            if ( ! $product ) {
                continue;
            }

            $weight = (float) $product->get_weight();
            if ( $weight <= 0 ) {
                $missing[] = array(
                    'id'   => $product_id,
                    'name' => $product->get_name(),
                );
            }
        }

        $products_without_weight = $missing;
        // Cache for 1 hour
        set_transient( 'atoship_products_missing_weight', $missing, HOUR_IN_SECONDS );
    }

    if ( empty( $products_without_weight ) ) {
        return;
    }

    $count = count( $products_without_weight );

    // Build product list (show up to 5)
    $product_links = array();
    $shown = array_slice( $products_without_weight, 0, 5 );
    foreach ( $shown as $p ) {
        $edit_url = get_edit_post_link( $p['id'] );
        $product_links[] = '<a href="' . esc_url( $edit_url ) . '">' . esc_html( $p['name'] ) . '</a>';
    }

    $list_html = implode( ', ', $product_links );
    if ( $count > 5 ) {
        /* translators: %d: number of additional products */
        $list_html .= sprintf( __( ' and %d more', 'atoship-for-woocommerce' ), $count - 5 );
    }

    printf(
        '<div class="notice notice-warning is-dismissible"><p><strong>%s</strong> %s</p><p>%s</p></div>',
        esc_html__( 'AtoShip Shipping:', 'atoship-for-woocommerce' ),
        sprintf(
            /* translators: %d: number of products without weight */
            esc_html( _n(
                '%d product does not have a weight set. Shipping rates may be inaccurate.',
                '%d products do not have a weight set. Shipping rates may be inaccurate.',
                $count,
                'atoship-for-woocommerce'
            ) ),
            $count
        ),
        $list_html
    );
}
add_action( 'admin_notices', 'atoship_check_products_missing_weight' );

/**
 * Clear the missing-weight notice cache when a product is saved
 */
function atoship_clear_weight_notice_cache( $product_id ) {
    delete_transient( 'atoship_products_missing_weight' );
}
add_action( 'woocommerce_update_product', 'atoship_clear_weight_notice_cache' );
add_action( 'woocommerce_new_product', 'atoship_clear_weight_notice_cache' );
