<?php
/**
 * AtoShip Webhook Handler
 *
 * Handles incoming webhooks from AtoShip
 *
 * @package AtoShip
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AtoShip_Webhook {

    /**
     * Supported webhook events
     */
    const SUPPORTED_EVENTS = array(
        'label.created',
        'label.purchased',
        'label.voided',
        'label.refunded',
        'tracking.created',
        'tracking.updated',
        'tracking.delivered',
        'tracking.exception',
    );

    /**
     * Initialize webhook handler
     */
    public static function init() {
        add_action( 'rest_api_init', array( __CLASS__, 'register_routes' ) );
    }

    /**
     * Register REST API routes
     */
    public static function register_routes() {
        register_rest_route( 'atoship/v1', '/webhook', array(
            'methods'             => 'POST',
            'callback'            => array( __CLASS__, 'handle_webhook' ),
            'permission_callback' => '__return_true', // Signature verification happens in handler
        ) );

        // Activate endpoint: atoship backend pushes OAuth token after WooCommerce OAuth 1.0a callback
        register_rest_route( 'atoship/v1', '/activate', array(
            'methods'             => 'POST',
            'callback'            => array( __CLASS__, 'handle_activate' ),
            'permission_callback' => array( __CLASS__, 'verify_wc_credentials' ),
        ) );

        // Disconnect endpoint: atoship notifies plugin when store is deleted
        register_rest_route( 'atoship/v1', '/disconnect', array(
            'methods'             => 'POST',
            'callback'            => array( __CLASS__, 'handle_disconnect' ),
            'permission_callback' => array( __CLASS__, 'verify_wc_credentials' ),
        ) );

        // Status endpoint: allows atoship to check if plugin is installed
        register_rest_route( 'atoship/v1', '/status', array(
            'methods'             => 'GET',
            'callback'            => array( __CLASS__, 'handle_status' ),
            'permission_callback' => '__return_true',
        ) );
    }

    /**
     * Verify WooCommerce consumer credentials (Basic Auth)
     *
     * @param WP_REST_Request $request Request object
     * @return bool
     */
    public static function verify_wc_credentials( $request ) {
        $auth_header = $request->get_header( 'Authorization' );

        if ( empty( $auth_header ) || 0 !== strpos( $auth_header, 'Basic ' ) ) {
            return false;
        }

        $decoded = base64_decode( substr( $auth_header, 6 ) );
        if ( false === $decoded || false === strpos( $decoded, ':' ) ) {
            return false;
        }

        list( $consumer_key, $consumer_secret ) = explode( ':', $decoded, 2 );

        if ( empty( $consumer_key ) || empty( $consumer_secret ) ) {
            return false;
        }

        // Look up the consumer key in WooCommerce API keys table
        // WooCommerce stores consumer_key hashed, consumer_secret in plain text
        global $wpdb;
        $key_data = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT consumer_secret, permissions FROM {$wpdb->prefix}woocommerce_api_keys WHERE consumer_key = %s",
                wc_api_hash( $consumer_key )
            )
        );

        if ( ! $key_data ) {
            return false;
        }

        // Verify consumer secret matches
        if ( ! hash_equals( $key_data->consumer_secret, $consumer_secret ) ) {
            return false;
        }

        return true;
    }

    /**
     * Handle activate request — save OAuth token pushed by atoship backend
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public static function handle_activate( $request ) {
        $body = $request->get_json_params();

        $access_token  = isset( $body['access_token'] ) ? sanitize_text_field( $body['access_token'] ) : '';
        $refresh_token = isset( $body['refresh_token'] ) ? sanitize_text_field( $body['refresh_token'] ) : '';
        $expires_in    = isset( $body['expires_in'] ) ? absint( $body['expires_in'] ) : 3600;
        $scope         = isset( $body['scope'] ) ? sanitize_text_field( $body['scope'] ) : '';

        if ( empty( $access_token ) ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => 'Missing access_token',
            ), 400 );
        }

        // Save the token via OAuth class
        AtoShip_OAuth::activate_with_token( $access_token, $refresh_token, $expires_in, $scope );

        self::log( 'AtoShip connection activated via backend token push' );

        return new WP_REST_Response( array(
            'success' => true,
            'message' => 'AtoShip connection activated',
            'plugin_version' => ATOSHIP_VERSION,
        ), 200 );
    }

    /**
     * Handle disconnect request — clear OAuth tokens when store is deleted on atoship
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public static function handle_disconnect( $request ) {
        AtoShip_OAuth::disconnect( false );

        self::log( 'AtoShip connection disconnected via backend notification' );

        return new WP_REST_Response( array(
            'success' => true,
            'message' => 'AtoShip connection disconnected',
        ), 200 );
    }

    /**
     * Handle status check — public endpoint for atoship to verify plugin is installed
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public static function handle_status( $request ) {
        return new WP_REST_Response( array(
            'success'        => true,
            'plugin'         => 'atoship-for-woocommerce',
            'version'        => ATOSHIP_VERSION,
            'connected'      => AtoShip_OAuth::is_connected(),
            'woocommerce'    => defined( 'WC_VERSION' ) ? WC_VERSION : null,
        ), 200 );
    }

    /**
     * Handle incoming webhook
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public static function handle_webhook( $request ) {
        // Get raw body for signature verification
        $raw_body = $request->get_body();
        $payload  = json_decode( $raw_body, true );

        // Get headers
        $signature   = $request->get_header( 'X-Atoship-Signature' );
        $event_type  = $request->get_header( 'X-Atoship-Event' );
        $delivery_id = $request->get_header( 'X-Atoship-Delivery' );

        // Log incoming webhook
        self::log( sprintf( 'Received webhook: %s (Delivery: %s)', $event_type, $delivery_id ) );

        // Verify signature
        $webhook_secret = get_option( 'atoship_webhook_secret', '' );
        if ( ! empty( $webhook_secret ) && ! empty( $signature ) ) {
            if ( ! AtoShip_API::verify_webhook_signature( $raw_body, $signature, $webhook_secret ) ) {
                self::log( 'Webhook signature verification failed', 'error' );
                return new WP_REST_Response( array(
                    'success' => false,
                    'message' => 'Invalid signature',
                ), 401 );
            }
        }

        // Validate payload
        if ( empty( $payload ) || ! isset( $payload['type'] ) ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => 'Invalid payload',
            ), 400 );
        }

        // Process the webhook
        $result = self::process_webhook( $payload );

        if ( is_wp_error( $result ) ) {
            self::log( sprintf( 'Webhook processing failed: %s', $result->get_error_message() ), 'error' );
            return new WP_REST_Response( array(
                'success' => false,
                'message' => $result->get_error_message(),
            ), 500 );
        }

        return new WP_REST_Response( array(
            'success' => true,
            'message' => 'Webhook processed',
        ), 200 );
    }

    /**
     * Process webhook based on event type
     *
     * @param array $payload Webhook payload
     * @return bool|WP_Error
     */
    private static function process_webhook( $payload ) {
        $event_type = $payload['type'];

        // Route to appropriate handler
        switch ( $event_type ) {
            case 'label.created':
            case 'label.purchased':
                return self::handle_label_created( $payload );

            case 'label.voided':
            case 'label.refunded':
                return self::handle_label_voided( $payload );

            case 'tracking.created':
            case 'tracking.updated':
                return self::handle_tracking_update( $payload );

            case 'tracking.delivered':
                return self::handle_tracking_delivered( $payload );

            case 'tracking.exception':
                return self::handle_tracking_exception( $payload );

            default:
                self::log( sprintf( 'Unhandled webhook event type: %s', $event_type ) );
                return true;
        }
    }

    /**
     * Handle label.created / label.purchased event
     *
     * @param array $payload Webhook payload
     * @return bool|WP_Error
     */
    private static function handle_label_created( $payload ) {
        $data = $payload['data']['object'] ?? array();

        // Find the WooCommerce order
        $order = self::find_order_by_atoship_id( $data['order_id'] ?? '' );
        if ( ! $order ) {
            $order = self::find_order_by_external_id( $data['external_id'] ?? '' );
        }

        if ( ! $order ) {
            return new WP_Error( 'order_not_found', 'Could not find associated WooCommerce order' );
        }

        // Save shipment data
        global $wpdb;
        $table_name = $wpdb->prefix . 'atoship_shipments';

        $wpdb->insert( $table_name, array(
            'order_id'            => $order->get_id(),
            'atoship_shipment_id' => $data['id'] ?? '',
            'carrier'             => $data['carrier'] ?? '',
            'tracking_number'     => $data['tracking_number'] ?? '',
            'tracking_url'        => $data['tracking_url'] ?? '',
            'label_url'           => $data['label_url'] ?? '',
            'status'              => 'label_created',
            'ship_date'           => isset( $data['ship_date'] ) ? date( 'Y-m-d H:i:s', strtotime( $data['ship_date'] ) ) : null,
        ) );

        // Update order meta
        $order->update_meta_data( '_atoship_tracking_number', $data['tracking_number'] ?? '' );
        $order->update_meta_data( '_atoship_carrier', $data['carrier'] ?? '' );
        $order->update_meta_data( '_atoship_tracking_url', $data['tracking_url'] ?? '' );
        $order->update_meta_data( '_atoship_label_url', $data['label_url'] ?? '' );
        $order->save();

        // Add order note
        $order->add_order_note(
            sprintf(
                /* translators: 1: Carrier name, 2: Tracking number */
                __( 'Shipping label created via AtoShip. Carrier: %1$s, Tracking: %2$s', 'atoship-for-woocommerce' ),
                $data['carrier'] ?? 'N/A',
                $data['tracking_number'] ?? 'N/A'
            ),
            false,
            true
        );

        // Send tracking email if enabled
        if ( 'yes' === get_option( 'atoship_send_tracking_email', 'yes' ) ) {
            self::send_tracking_email( $order, $data );
        }

        self::log( sprintf( 'Label created for order #%d. Tracking: %s', $order->get_id(), $data['tracking_number'] ?? 'N/A' ) );

        return true;
    }

    /**
     * Handle label.voided / label.refunded event
     *
     * @param array $payload Webhook payload
     * @return bool|WP_Error
     */
    private static function handle_label_voided( $payload ) {
        $data = $payload['data']['object'] ?? array();

        // Update shipment status
        global $wpdb;
        $table_name = $wpdb->prefix . 'atoship_shipments';

        $wpdb->update(
            $table_name,
            array( 'status' => 'voided' ),
            array( 'atoship_shipment_id' => $data['id'] ?? '' )
        );

        // Find and update order
        $order = self::find_order_by_atoship_id( $data['order_id'] ?? '' );
        if ( $order ) {
            $order->add_order_note(
                __( 'Shipping label voided via AtoShip.', 'atoship-for-woocommerce' ),
                false,
                true
            );
        }

        return true;
    }

    /**
     * Handle tracking.created / tracking.updated event
     *
     * @param array $payload Webhook payload
     * @return bool|WP_Error
     */
    private static function handle_tracking_update( $payload ) {
        $data = $payload['data']['object'] ?? array();

        // Find the order
        $order = self::find_order_by_tracking( $data['tracking_number'] ?? '' );
        if ( ! $order ) {
            return new WP_Error( 'order_not_found', 'Could not find order for tracking number' );
        }

        // Update tracking status
        $order->update_meta_data( '_atoship_tracking_status', $data['status'] ?? '' );
        $order->update_meta_data( '_atoship_tracking_updated', current_time( 'mysql' ) );

        // Save tracking events if provided
        if ( ! empty( $data['events'] ) ) {
            $order->update_meta_data( '_atoship_tracking_events', $data['events'] );
        }

        $order->save();

        // Add order note for significant status changes
        $status_message = $data['status_description'] ?? $data['status'] ?? '';
        if ( ! empty( $status_message ) ) {
            $order->add_order_note(
                sprintf(
                    /* translators: %s: Tracking status */
                    __( 'Tracking update: %s', 'atoship-for-woocommerce' ),
                    $status_message
                ),
                false,
                true
            );
        }

        return true;
    }

    /**
     * Handle tracking.delivered event
     *
     * @param array $payload Webhook payload
     * @return bool|WP_Error
     */
    private static function handle_tracking_delivered( $payload ) {
        $data = $payload['data']['object'] ?? array();

        // Find the order
        $order = self::find_order_by_tracking( $data['tracking_number'] ?? '' );
        if ( ! $order ) {
            return new WP_Error( 'order_not_found', 'Could not find order for tracking number' );
        }

        // Update meta
        $order->update_meta_data( '_atoship_tracking_status', 'delivered' );
        $order->update_meta_data( '_atoship_delivered_at', $data['delivered_at'] ?? current_time( 'mysql' ) );
        $order->save();

        // Add order note
        $order->add_order_note(
            __( 'Package delivered! (via AtoShip tracking)', 'atoship-for-woocommerce' ),
            true, // Notify customer
            true
        );

        // Auto-complete order if enabled
        if ( 'yes' === get_option( 'atoship_auto_complete', 'no' ) ) {
            if ( $order->get_status() !== 'completed' ) {
                $order->update_status( 'completed', __( 'Order auto-completed: package delivered.', 'atoship-for-woocommerce' ) );
            }
        }

        self::log( sprintf( 'Order #%d marked as delivered.', $order->get_id() ) );

        return true;
    }

    /**
     * Handle tracking.exception event
     *
     * @param array $payload Webhook payload
     * @return bool|WP_Error
     */
    private static function handle_tracking_exception( $payload ) {
        $data = $payload['data']['object'] ?? array();

        // Find the order
        $order = self::find_order_by_tracking( $data['tracking_number'] ?? '' );
        if ( ! $order ) {
            return new WP_Error( 'order_not_found', 'Could not find order for tracking number' );
        }

        // Update meta
        $order->update_meta_data( '_atoship_tracking_status', 'exception' );
        $order->update_meta_data( '_atoship_tracking_exception', $data['exception_description'] ?? '' );
        $order->save();

        // Add order note (important - notify admin)
        $order->add_order_note(
            sprintf(
                /* translators: %s: Exception description */
                __( 'Tracking exception: %s', 'atoship-for-woocommerce' ),
                $data['exception_description'] ?? __( 'Unknown exception', 'atoship-for-woocommerce' )
            ),
            false,
            true
        );

        self::log( sprintf( 'Tracking exception for order #%d: %s', $order->get_id(), $data['exception_description'] ?? '' ), 'warning' );

        return true;
    }

    /**
     * Find WooCommerce order by AtoShip order ID
     *
     * @param string $atoship_order_id AtoShip order ID
     * @return WC_Order|null
     */
    private static function find_order_by_atoship_id( $atoship_order_id ) {
        if ( empty( $atoship_order_id ) ) {
            return null;
        }

        $orders = wc_get_orders( array(
            'meta_key'   => '_atoship_order_id',
            'meta_value' => $atoship_order_id,
            'limit'      => 1,
        ) );

        return ! empty( $orders ) ? $orders[0] : null;
    }

    /**
     * Find WooCommerce order by external ID (WC order ID)
     *
     * @param string $external_id External ID (WooCommerce order ID)
     * @return WC_Order|null
     */
    private static function find_order_by_external_id( $external_id ) {
        if ( empty( $external_id ) ) {
            return null;
        }

        return wc_get_order( absint( $external_id ) );
    }

    /**
     * Find WooCommerce order by tracking number
     *
     * @param string $tracking_number Tracking number
     * @return WC_Order|null
     */
    private static function find_order_by_tracking( $tracking_number ) {
        if ( empty( $tracking_number ) ) {
            return null;
        }

        $orders = wc_get_orders( array(
            'meta_key'   => '_atoship_tracking_number',
            'meta_value' => $tracking_number,
            'limit'      => 1,
        ) );

        return ! empty( $orders ) ? $orders[0] : null;
    }

    /**
     * Send tracking email to customer
     *
     * @param WC_Order $order Order object
     * @param array    $data  Shipment data
     */
    private static function send_tracking_email( $order, $data ) {
        $mailer = WC()->mailer();

        $recipient = $order->get_billing_email();
        $subject   = sprintf(
            /* translators: %s: Order number */
            __( 'Your order #%s has been shipped!', 'atoship-for-woocommerce' ),
            $order->get_order_number()
        );

        $tracking_url = $data['tracking_url'] ?? '';
        $carrier      = $data['carrier'] ?? '';
        $tracking_no  = $data['tracking_number'] ?? '';

        ob_start();
        ?>
        <p><?php printf( esc_html__( 'Hi %s,', 'atoship-for-woocommerce' ), esc_html( $order->get_billing_first_name() ) ); ?></p>
        <p><?php esc_html_e( 'Great news! Your order has been shipped.', 'atoship-for-woocommerce' ); ?></p>
        <p>
            <strong><?php esc_html_e( 'Carrier:', 'atoship-for-woocommerce' ); ?></strong> <?php echo esc_html( $carrier ); ?><br>
            <strong><?php esc_html_e( 'Tracking Number:', 'atoship-for-woocommerce' ); ?></strong> <?php echo esc_html( $tracking_no ); ?>
        </p>
        <?php if ( ! empty( $tracking_url ) ) : ?>
        <p>
            <a href="<?php echo esc_url( $tracking_url ); ?>" style="background-color: #0073aa; color: #ffffff; padding: 10px 20px; text-decoration: none; border-radius: 3px; display: inline-block;">
                <?php esc_html_e( 'Track Your Package', 'atoship-for-woocommerce' ); ?>
            </a>
        </p>
        <?php endif; ?>
        <p><?php esc_html_e( 'Thank you for your order!', 'atoship-for-woocommerce' ); ?></p>
        <?php
        $message = ob_get_clean();

        $headers = array( 'Content-Type: text/html; charset=UTF-8' );

        $mailer->send( $recipient, $subject, $message, $headers );
    }

    /**
     * Log message
     *
     * @param string $message Message to log
     * @param string $level   Log level
     */
    private static function log( $message, $level = 'info' ) {
        if ( function_exists( 'wc_get_logger' ) ) {
            $logger = wc_get_logger();
            $logger->log( $level, '[Webhook] ' . $message, array( 'source' => 'atoship' ) );
        }
    }
}
